package tiroleinternal

import (
	"fmt"
	"net/http"
	"strings"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tirole_internal/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/tirole_internal/internal/reqs"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

func (t *TiroleInternal) HandleManageSlug() echo.HandlerFunc {
	return func(c echo.Context) error {
		req, err := ParseManageSlugRequest(c)
		if err != nil {
			return t.sendErrorResponse(c, err)
		}

		ctxlog.Debugf(c.Request().Context(), logger.Log(),
			"manage_slug: trying to update slug='%s', tvmid=%s", req.SystemSlug, printTvmID(req.Tvmid))

		if err := t.yt.ManageSlug(c.Request().Context(), req); err != nil {
			return t.sendErrorResponse(c, err)
		}

		return c.JSON(http.StatusOK, SimpleResponse{Status: "ok"})
	}
}

func ParseManageSlugRequest(c echo.Context) (*reqs.ManageSlug, error) {
	contentType := c.Request().Header.Get(echo.HeaderContentType)
	if !strings.HasPrefix(contentType, echo.MIMEApplicationJSON) {
		return nil, &errs.InvalidRequestError{
			Message: fmt.Sprintf(
				"Only JSON allowed as request, got Content-Type: '%s'", contentType),
		}
	}

	res := &reqs.ManageSlug{}
	if err := (&echo.DefaultBinder{}).BindBody(c, &res); err != nil {
		return nil, &errs.InvalidRequestError{
			Message: fmt.Sprintf("Bad JSON body: %s", err.Error()),
		}
	}

	if res.SystemSlug == "" {
		return nil, &errs.InvalidRequestError{
			Message: "'system_slug' cannot be empty",
		}
	}
	for _, t := range res.Tvmid {
		if t == 0 {
			return nil, &errs.InvalidRequestError{
				Message: "Tvmid cannot be 0",
			}
		}
	}

	return res, nil
}

func printTvmID(tvmid []tvm.ClientID) string {
	tmp := make([]string, len(tvmid))

	for idx := range tvmid {
		tmp[idx] = fmt.Sprintf("%d", tvmid[idx])
	}

	return strings.Join(tmp, ",")
}
