package tiroleinternal

import (
	"bytes"
	"crypto/hmac"
	"crypto/sha256"
	"encoding/base32"
	"encoding/hex"
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"strings"
	"time"

	"github.com/andybalholm/brotli"
	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/tirole_internal/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/tirole_internal/internal/reqs"
	"a.yandex-team.ru/passport/infra/daemons/tirole_internal/internal/ytc"
	"a.yandex-team.ru/passport/infra/daemons/tirole_internal/keys"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

func (t *TiroleInternal) HandleUploadRoles() echo.HandlerFunc {
	return func(c echo.Context) error {
		req, err := ParseUploadRolesRequest(c)
		if err != nil {
			return t.sendErrorResponse(c, err)
		}

		ctxlog.Debugf(c.Request().Context(), logger.Log(),
			"upload_roles: trying to insert role for slug='%s', revision=%d", req.SystemSlug, req.Roles.Revision)

		prepared, err := PrepareYtReq(req, t.keyMap)
		if err != nil {
			return t.sendErrorResponse(c, err)
		}
		if err := t.yt.UploadRoles(c.Request().Context(), prepared); err != nil {
			return t.sendErrorResponse(c, err)
		}

		return c.JSON(http.StatusOK, SimpleResponse{Status: "ok"})
	}
}

func ParseUploadRolesRequest(c echo.Context) (*reqs.UploadRoles, error) {
	contentType := c.Request().Header.Get(echo.HeaderContentType)
	if !strings.HasPrefix(contentType, echo.MIMEApplicationJSON) {
		return nil, &errs.InvalidRequestError{
			Message: fmt.Sprintf("Only JSON allowed as request, got Content-Type: '%s'", contentType),
		}
	}

	body, err := io.ReadAll(c.Request().Body)
	if err != nil {
		return nil, &errs.TemporaryError{
			Message: fmt.Sprintf("Failed to fetch request body: '%s'", err.Error()),
		}
	}

	var v interface{}
	err = json.Unmarshal(body, &v)
	if err != nil {
		return nil, &errs.InvalidRequestError{
			Message: fmt.Sprintf("Failed to unmarshal request body as JSON:\n%#v", err),
		}
	}

	err = reqs.UploadRolesSchema.Validate(v)
	if err != nil {
		return nil, &errs.InvalidRequestError{
			Message: fmt.Sprintf("Failed to validate request body:\n%#v", err),
		}
	}

	res := &reqs.UploadRoles{}
	if err := json.Unmarshal(body, res); err != nil {
		// unreachable now
		return nil, &errs.InvalidRequestError{
			Message: fmt.Sprintf("Failed to parse body: '%s'", err),
		}
	}

	return res, nil
}

func PrepareYtReq(req *reqs.UploadRoles, keyMap *keys.KeyMap) (*ytc.UploadRolesReq, error) {
	roles := castToRolesExt(&req.Roles)

	blob, err := json.Marshal(roles)
	if err != nil {
		return nil, xerrors.Errorf("failed to serialize roles: %w", err)
	}

	compressed, codec, err := compressBlob(blob)
	if err != nil {
		return nil, err
	}

	hash := hmac.New(sha256.New, keyMap.GetDefaultKey())
	hash.Write(compressed)
	sign := hash.Sum(nil)

	timeZone, _ := time.LoadLocation("Europe/Moscow")

	res := &ytc.UploadRolesReq{
		Slug:     req.SystemSlug,
		Revision: reverseRevision(req.Roles.Revision),
		Meta: ytc.UploadRolesMetaReq{
			Unixtime:      req.Roles.BornDate,
			Codec:         codec,
			DecodedSize:   uint64(len(blob)),
			DecodedSha256: fmt.Sprintf("%X", sha256.Sum256(blob)),
			EncodedHmac:   fmt.Sprintf("%s:%s", keyMap.DefaultKeyID, hex.EncodeToString(sign)),
			RevisionExt:   roles.RevisionExt,
			Revision:      req.Roles.Revision,
			Borndate:      time.Unix(int64(req.Roles.BornDate), 0).In(timeZone).String(),
		},
		Blob: compressed,
	}

	return res, nil
}

func compressBlob(blob []byte) ([]byte, string, error) {
	buffer := new(bytes.Buffer)
	gw := brotli.NewWriterLevel(buffer, 8)

	_, err := gw.Write(blob)
	if err != nil {
		return nil, "", xerrors.Errorf("failed to compress roles: %w", err)
	}
	if err := gw.Close(); err != nil {
		return nil, "", xerrors.Errorf("failed to finish compressing of roles: %w", err)
	}

	return buffer.Bytes(), "brotli", nil
}

func reverseRevision(rev uint64) uint64 {
	return uint64(0x7fffffffffffffff) - rev
}

func castToRolesExt(req *reqs.Roles) *reqs.RolesExt {
	return &reqs.RolesExt{
		RevisionExt: makeRevisionExternal(req.Revision),
		BornDate:    req.BornDate,
		Tvm:         req.Tvm,
		User:        req.User,
	}
}

func makeRevisionExternal(rev uint64) string {
	encoder := base32.StdEncoding.WithPadding(base32.NoPadding)
	return encoder.EncodeToString([]byte(fmt.Sprintf("%x", rev)))
}
