package tiroleinternal

import (
	"fmt"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tirole_internal/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/tirole_internal/internal/ytc"
	"a.yandex-team.ru/passport/infra/daemons/tirole_internal/keys"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/httpdtvm"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/middlewares"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/passport/shared/golibs/unistat"
)

type TiroleInternal struct {
	tvm       tvm.Client
	yt        *ytc.Client
	cfg       Config
	accessLog log.Logger
	unistat   stats
	keyMap    *keys.KeyMap
}

type Config struct {
	Common       CommonConfig       `json:"common"`
	Tvm          httpdtvm.TvmConfig `json:"tvm"`
	AllowedTvmid []tvm.ClientID     `json:"allowed_tvmid"`
	Yt           ytc.Config         `json:"yt"`
	KeyMap       keys.Config        `json:"key_map"`
}

type CommonConfig struct {
	ForceDownFile string `json:"force_down_file"`
	AccessLog     string `json:"access_log"`
}

type stats struct {
	errInvalidRequest *unistat.SignalDiff
	errTmp            *unistat.SignalDiff
	errUnauthorized   *unistat.SignalDiff
	errUnknown        *unistat.SignalDiff
}

type Factory struct{}

func (f *Factory) NewService(config httpdaemon.ServiceConfig) (httpdaemon.Service, error) {
	var cfg Config
	if err := httpdaemon.ParseServiceConfig(config, &cfg); err != nil {
		return nil, err
	}

	tvmClient, err := httpdtvm.InitTvm(cfg.Tvm)
	if err != nil {
		return nil, err
	}

	keyMap, err := keys.InitKeyMap(cfg.KeyMap)
	if err != nil {
		return nil, err
	}

	ytClient, err := ytc.InitYt(cfg.Yt)
	if err != nil {
		return nil, err
	}

	access, err := logger.CreateLog(logger.Config{
		FilePath:             cfg.Common.AccessLog,
		DisablePrintingLevel: true,
	})
	if err != nil {
		return nil, err
	}

	res := &TiroleInternal{
		tvm:       tvmClient,
		yt:        ytClient,
		cfg:       cfg,
		accessLog: access,
		unistat: stats{
			errInvalidRequest: unistat.DefaultChunk.CreateSignalDiff("errors.requests.invalid_request"),
			errTmp:            unistat.DefaultChunk.CreateSignalDiff("errors.requests.tmp_err"),
			errUnauthorized:   unistat.DefaultChunk.CreateSignalDiff("errors.requests.unauthorized"),
			errUnknown:        unistat.DefaultChunk.CreateSignalDiff("errors.requests.unknown"),
		},
		keyMap: keyMap,
	}
	return res, nil
}

func (t *TiroleInternal) AddHandlers(e *echo.Echo) {
	e.Pre(
		t.middlewareAccessLog(),
		t.middlewareSendReqID(),
	)

	e.GET(
		"/ping",
		t.HandlePing(),
	)
	e.POST(
		"/v1/upload_roles",
		t.HandleUploadRoles(),
		t.middlewareTvmAuth(),
	)
	e.POST(
		"/v1/manage_slug",
		t.HandleManageSlug(),
		t.middlewareTvmAuth(),
	)
}

const emtryString = "-"

func (t *TiroleInternal) middlewareAccessLog() echo.MiddlewareFunc {
	nullable := func(val string) string {
		if len(val) == 0 {
			return emtryString
		}
		return val
	}

	consumer := func(t *tvm.CheckedServiceTicket) string {
		if t == nil {
			return emtryString
		}
		return fmt.Sprintf("%d", t.SrcID)
	}

	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			c.Response().After(func() {
				startTime := middlewares.ContextStartInstant(c.Request().Context())
				st := tvm.ContextServiceTicket(c.Request().Context())

				t.accessLog.Debugf(
					"%s\t%s\t%s\t%s\t%.1fms\t%d\t%d\t%s",
					middlewares.ContextReqID(c.Request().Context()),
					c.RealIP(),
					consumer(st),
					c.Request().URL.Path,
					float64(time.Since(startTime).Microseconds())/1000,
					c.Response().Status,
					c.Response().Size,
					nullable(c.Request().URL.RawQuery),
				)
			})

			return next(c)
		}
	}
}

func (t *TiroleInternal) middlewareSendReqID() echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			c.Response().Header().Set(
				echo.HeaderXRequestID,
				middlewares.ContextReqID(c.Request().Context()),
			)

			return next(c)
		}
	}
}

func (t *TiroleInternal) middlewareTvmAuth() echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			st := c.Request().Header.Get("X-Ya-Service-Ticket")
			if st == "" {
				return t.sendErrorResponse(c, &errs.UnauthorizedError{
					Message: "missing service ticket",
				})
			}

			ticket, err := t.tvm.CheckServiceTicket(c.Request().Context(), st)
			if err != nil {
				e := &errs.UnauthorizedError{
					Message:      "service ticket is invalid",
					Description:  err.Error(),
					TicketStatus: err.(*tvm.TicketError).Status.String(),
				}
				if ticket != nil {
					e.LoggablePart = ticket.LogInfo
				}
				if err.(*tvm.TicketError).Status == tvm.TicketInvalidDst {
					e.Description = fmt.Sprintf(
						"%s; expected dst is %d",
						e.Description, t.cfg.Tvm.SelfID,
					)
				}

				return t.sendErrorResponse(c, e)
			}

			c.SetRequest(c.Request().WithContext(
				tvm.WithServiceTicket(c.Request().Context(), ticket),
			))

			for _, i := range t.cfg.AllowedTvmid {
				if i == ticket.SrcID {
					return next(c)
				}
			}

			return t.sendErrorResponse(c, &errs.UnauthorizedError{
				Message: fmt.Sprintf("SrcID %d is not allowed", ticket.SrcID),
			})
		}
	}
}
