package ytc

import (
	"context"
	"fmt"
	"strconv"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/tirole_internal/internal/reqs"
	"a.yandex-team.ru/passport/infra/libs/go/ytsimple"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
)

type mappingRow struct {
	Key   string `yson:"slug"`
	TvmID uint32 `yson:"tvm_id"`
}

func (c *Client) ManageSlug(ctx context.Context, req *reqs.ManageSlug) error {
	var err error

	start := time.Now()
	defer func() {
		c.unistat.responseTimings.Insert(time.Since(start))
		if err != nil {
			c.unistat.errs.Inc()
		}
	}()
	c.unistat.requests.Inc()

	transaction, err := c.yc.BeginTabletTx(ctx, nil)
	if err != nil {
		return xerrors.Errorf("ManageSlug: failed to create transaction: %w", err)
	}

	currentMapping, err := c.getCurrentMapping(ctx, transaction, req.SystemSlug)
	if err != nil {
		return xerrors.Errorf("ManageSlug: failed to get current mapping: %w", err)
	}
	if err = c.deleteCurrentMapping(ctx, transaction, currentMapping); err != nil {
		return xerrors.Errorf("ManageSlug: failed to delete current mapping: %w", err)
	}
	if err = c.insertNewMapping(ctx, transaction, req); err != nil {
		return xerrors.Errorf("ManageSlug: failed to insert new mapping: %w", err)
	}

	if err = transaction.Commit(); err != nil {
		return xerrors.Errorf(
			"ManageSlug: failed to commit transaction for slug: %w",
			err,
		)
	}

	return err
}

func (c *Client) getCurrentMapping(ctx context.Context, tx yt.TabletTx, slug string) ([]interface{}, error) {
	query := fmt.Sprintf(
		"slug, tvm_id FROM [%s/mapping] WHERE slug=%s",
		c.path,
		strconv.Quote(slug),
	)

	res := make([]interface{}, 0)
	err := ytsimple.SelectAll(ctx, tx, query, c.timeout, func(reader yt.TableReader) error {
		var row mappingRow
		if err := ytsimple.ScanRow(reader, &row); err != nil {
			return err
		}

		res = append(res, &row)
		return nil
	})

	return res, err
}

func (c *Client) deleteCurrentMapping(ctx context.Context, tx yt.TabletTx, cur []interface{}) error {
	if len(cur) == 0 {
		return nil
	}

	path := ypath.Path(fmt.Sprintf("%s/mapping", c.path))

	requireSyncReplica := true
	options := &yt.DeleteRowsOptions{
		RequireSyncReplica: &requireSyncReplica,
	}

	if err := tx.DeleteRows(ctx, path, cur, options); err != nil {
		return xerrors.Errorf("ManageSlug: failed to delete rows: %w", err)
	}

	return nil
}

func (c *Client) insertNewMapping(ctx context.Context, tx yt.TabletTx, req *reqs.ManageSlug) error {
	if len(req.Tvmid) == 0 {
		return nil
	}

	path := ypath.Path(fmt.Sprintf("%s/mapping", c.path))

	type toInsert struct {
		Key   string `yson:"slug"`
		TvmID uint32 `yson:"tvm_id"`
	}
	keysToInsert := make([]interface{}, len(req.Tvmid))
	for idx, t := range req.Tvmid {
		keysToInsert[idx] = &toInsert{
			Key:   req.SystemSlug,
			TvmID: uint32(t),
		}
	}

	requireSyncReplica := true
	atomicity := yt.AtomicityFull
	options := &yt.InsertRowsOptions{
		RequireSyncReplica: &requireSyncReplica,
		Atomicity:          &atomicity,
	}

	if err := tx.InsertRows(ctx, path, keysToInsert, options); err != nil {
		return xerrors.Errorf(
			"ManageSlug: failed to insert rows: %w",
			err,
		)
	}

	return nil
}
