package ytc

import (
	"context"
	"io/ioutil"
	"time"

	"golang.org/x/xerrors"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/passport/shared/golibs/unistat"
	"a.yandex-team.ru/yt/go/yson"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
)

type Config struct {
	Cluster        string `json:"cluster"`
	Path           string `json:"path"`
	OAuthTokenFile string `json:"oauth_token_file"`
	Timeout        uint64 `json:"timeout_ms"`
	UseTLS         bool   `json:"use_tls"`
}

type Client struct {
	yc      yt.Client
	timeout yson.Duration
	path    string
	unistat stats
}

type stats struct {
	responseTimings *unistat.TimeStat
	requests        *unistat.SignalDiff
	errs            *unistat.SignalDiff
}

type PermanentError struct {
	err error
}

func (e *PermanentError) Error() string {
	return e.err.Error()
}

func IsPermanent(err error) bool {
	_, ok := err.(*PermanentError)
	return ok
}

func InitYt(cfg Config) (*Client, error) {
	cl, err := CreateLowLevelClient(&cfg)
	if err != nil {
		return nil, err
	}

	responseTimings, err := unistat.DefaultChunk.CreateTimeStats(
		"yt.response_time",
		unistat.CreateTimeBoundsFromMaxValue(10*time.Second),
	)
	if err != nil {
		return nil, xerrors.Errorf("Failed to create time stats: %w", err)
	}

	return &Client{
		yc:      cl,
		timeout: yson.Duration(time.Duration(cfg.Timeout) * time.Millisecond),
		path:    cfg.Path,
		unistat: stats{
			responseTimings: responseTimings,
			requests:        unistat.DefaultChunk.CreateSignalDiff("yt.requests"),
			errs:            unistat.DefaultChunk.CreateSignalDiff("yt.errors"),
		},
	}, nil
}

func CreateLowLevelClient(cfg *Config) (yt.Client, error) {
	token, err := ioutil.ReadFile(cfg.OAuthTokenFile)
	if err != nil {
		return nil, xerrors.Errorf("Failed to read oauth token: %w: %s", err, cfg.OAuthTokenFile)
	}

	cl, err := ythttp.NewClient(&yt.Config{
		Proxy:            cfg.Cluster,
		UseTLS:           cfg.UseTLS,
		ProxyRole:        "passport",
		Token:            string(token),
		Logger:           logger.Log().With(log.Nil("YtClient")).Structured(),
		CompressionCodec: yt.ClientCodecGZIP,
	})
	if err != nil {
		return nil, xerrors.Errorf("Failed to create YT client: %w", err)
	}

	return cl, nil
}

func (c *Client) createSelector(ctx context.Context, t yt.TabletClient, query string) (yt.TableReader, error) {
	options := &yt.SelectRowsOptions{}
	options.TimeoutOptions = &yt.TimeoutOptions{
		Timeout: c.timeout,
	}

	reader, err := t.SelectRows(ctx, query, options)
	if err != nil {
		return nil, xerrors.Errorf("failed to get create reader from YT: %w", err)
	}

	return reader, nil
}
