package keys

import (
	"encoding/hex"
	"encoding/json"
	"io/ioutil"

	"golang.org/x/xerrors"
)

type Config struct {
	Filename   string `json:"keys_file"`
	DefaultKey string `json:"default_key"`
}

type KeyMap struct {
	keys         map[string][]byte
	DefaultKeyID string
}

func InitKeyMap(cfg Config) (*KeyMap, error) {
	if cfg.Filename == "" || cfg.DefaultKey == "" {
		return nil, xerrors.Errorf("Keys file or default key not configured")
	}

	jsonFile, err := ioutil.ReadFile(cfg.Filename)
	if err != nil {
		return nil, xerrors.Errorf("Failed to read keys file '%s': %w", cfg.Filename, err)
	}

	var keys map[string]string
	err = json.Unmarshal(jsonFile, &keys)
	if err != nil {
		return nil, xerrors.Errorf("Failed to parse keys file %s", cfg.Filename)
	}

	_, defaultFound := keys[cfg.DefaultKey]
	if !defaultFound {
		return nil, xerrors.Errorf("Default key '%s' not in keys file %s", cfg.DefaultKey, cfg.Filename)
	}

	var keyMap KeyMap
	keyMap.keys = make(map[string][]byte)
	keyMap.DefaultKeyID = cfg.DefaultKey

	for id, body := range keys {
		if id == "" || body == "" {
			return nil, xerrors.Errorf("Empty key id or body: '%s':'%s'", id, body)
		}
		err = keyMap.AddHexKey(id, body)
		if err != nil {
			return nil, err
		}
	}

	return &keyMap, nil
}

func CreateKeyMap(defaultID string, defaultKeyHex string) (*KeyMap, error) {
	if defaultID == "" || defaultKeyHex == "" {
		return nil, xerrors.Errorf("Empty default key id or body: '%s':'%s'", defaultID, defaultKeyHex)
	}
	keyMap := new(KeyMap)

	keyMap.keys = make(map[string][]byte)
	keyMap.DefaultKeyID = defaultID
	err := keyMap.AddHexKey(defaultID, defaultKeyHex)
	if err != nil {
		return nil, err
	}

	return keyMap, nil
}

func (m *KeyMap) GetDefaultKey() []byte {
	return m.keys[m.DefaultKeyID]
}

func (m *KeyMap) GetKey(id string) []byte {
	key, found := m.keys[id]
	if found {
		return key
	}
	return nil
}

func (m *KeyMap) AddHexKey(id string, key string) error {
	bytes, err := hex.DecodeString(key)
	if err != nil || len(bytes) == 0 {
		return xerrors.Errorf("Bad hex key '%s'", key)
	}
	m.keys[id] = bytes
	return nil
}
