package updaterows

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"os"

	"github.com/andybalholm/brotli"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/infra/daemons/tirole_internal/internal/reqs"
	"a.yandex-team.ru/passport/infra/daemons/tirole_internal/internal/tiroleinternal"
	"a.yandex-team.ru/passport/infra/daemons/tirole_internal/internal/ytc"
	"a.yandex-team.ru/passport/infra/daemons/tirole_internal/keys"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
)

func tryUpdate(client yt.Client, keyMap *keys.KeyMap, row *ytc.UploadRolesReq, dir string) error {
	req := covertRowToReq(row)
	if req == nil {
		_, _ = fmt.Fprintf(os.Stderr, "====== %s;%d - skipped\n", row.Slug, row.Revision)
		return nil
	}

	prepared, err := tiroleinternal.PrepareYtReq(req, keyMap)
	if err != nil {
		_, _ = fmt.Fprintf(os.Stderr, "====== %s;%d - skipped: err: %s\n", row.Slug, row.Revision, err)
		return nil
	}

	if err := insert(client, prepared, dir); err != nil {
		_, _ = fmt.Fprintf(os.Stderr, "====== %s;%d - fail: %s\n", row.Slug, row.Revision, err)
		return err
	}

	_, _ = fmt.Fprintf(os.Stderr, "====== %s;%d - updated\n", row.Slug, row.Revision)
	return nil
}

func covertRowToReq(row *ytc.UploadRolesReq) *reqs.UploadRoles {
	decoded := make([]byte, row.Meta.DecodedSize)
	r := brotli.NewReader(bytes.NewBuffer(row.Blob))
	_, err := io.ReadFull(r, decoded)
	if err != nil {
		panic(err)
	}

	res := &reqs.UploadRoles{
		SystemSlug: row.Slug,
	}
	if err := json.Unmarshal(decoded, &res.Roles); err != nil {
		logger.Log().Warnf("=== %s", err)
		return nil
	}

	return res
}

func insert(client yt.Client, req *ytc.UploadRolesReq, dir string) error {
	path := ypath.Path(fmt.Sprintf("%s/roles", dir))

	requireSyncReplica := true
	atomicity := yt.AtomicityFull
	options := &yt.InsertRowsOptions{
		RequireSyncReplica: &requireSyncReplica,
		Atomicity:          &atomicity,
	}

	if err := client.InsertRows(context.Background(), path, []interface{}{req}, options); err != nil {
		return xerrors.Errorf(
			"UploadRoles: failed to insert row: %w",
			err,
		)
	}

	return nil
}
