#include "disk_cache.h"

#include <passport/infra/daemons/tvmapi/src/proto/pregen_raw_list.pb.h>
#include <passport/infra/daemons/tvmapi/src/utils/utils.h>

#include <passport/infra/libs/cpp/utils/log/global.h>

#include <library/cpp/tvmauth/src/utils.h>

#include <util/generic/maybe.h>
#include <util/stream/file.h>
#include <util/system/fs.h>

namespace NPassport::NTvm::NPregen {
    TDiskCache::TDiskCache(const TString& filename, ui32 keyTtl, ui32 minTicketTtl)
        : Filename_(filename)
        , KeyTtl_(keyTtl)
        , MinTicketTtl_(minTicketTtl)
    {
        if (NFs::Exists(Filename_)) {
            Init(TFileInput(Filename_).ReadAll());
            TLog::Info() << "Pregeneration: from " << Filename_ << " got "
                         << InitialValue_.Keys.size() << " keys, "
                         << InitialValue_.Tickets.size() << " tickets";
        }
    }

    void TDiskCache::WriteCache(const TKeyList::TData& keys, const TStorage::TData& tickets, const time_t now) const {
        TString body = ToArray(keys, tickets, now);
        if (body) {
            TUtils::WriteFileViaTmp(Filename_, body);
        }
    }

    TDiskCache TDiskCache::FromMemory(const TString& body, ui32 keyTtl, ui32 minTicketTtl, const time_t now) {
        TDiskCache res("", keyTtl, minTicketTtl);
        res.Init(body, now);
        return res;
    }

    TString TDiskCache::ToArray(const TKeyList::TData& keys, const TStorage::TData& tickets, const time_t now) const {
        pregen_raw_list::Data proto;
        TKeyList::ProcessData(
            keys,
            KeyTtl_,
            [&tickets, &proto](const TKey& k, time_t lastTime) {
                pregen_raw_list::Row* row = proto.add_row();
                row->set_dst(k.Dst);
                row->set_src(k.Src);
                row->set_last_time(lastTime);

                auto it = tickets.find(k);
                if (it != tickets.end()) {
                    row->set_ticket(it->second);
                }
            },
            now);

        return proto.SerializeAsString();
    }

    void TDiskCache::Init(const TString& body, const time_t now) {
        pregen_raw_list::Data proto;
        if (body.empty() || !proto.ParseFromString(body)) {
            return;
        }

        for (int idx = 0; idx < proto.row_size(); ++idx) {
            const pregen_raw_list::Row& r = proto.row(idx);
            const time_t lastTime = r.last_time();

            if (lastTime + KeyTtl_ <= now) {
                continue;
            }

            const TKey key{r.src(), r.dst()};
            InitialValue_.Keys.insert({key, std::make_shared<std::atomic<time_t>>(lastTime)});

            if (!r.has_ticket()) {
                continue;
            }

            const TString& ticket = r.ticket();

            TMaybe<TInstant> expTime = NTvmAuth::NInternal::TCanningKnife::GetExpirationTime(ticket);
            if (expTime && now + MinTicketTtl_ < expTime->TimeT()) {
                InitialValue_.Tickets.insert({key, ticket});
            }
        }
    }
}
