#include "facade.h"

#include "generator.h"

#include <passport/infra/daemons/tvmapi/src/runtime_context/runtime_context.h>
#include <passport/infra/daemons/tvmapi/src/utils/utils.h>

#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/file.h>

#include <util/stream/format.h>

namespace NPassport::NTvm::NPregen {
    static NUnistat::TTimeStat::TBounds CreateBounds() {
        NUnistat::TTimeStat::TBounds res;
        for (size_t idx = 0; idx < 50; ++idx) {
            res.push_back(TDuration::Seconds(10 * idx));
        }
        return res;
    }

    TFacade::TFacade(const TRuntimeContext& runtime)
        : Runtime_(runtime)
        , Generator_(std::make_unique<TGenerator>(TGenerator::TSettings{
              .KeyTtl = Runtime_.Config().Pregen.KeyTtl,
              .TicketTtl = Runtime_.Config().Gen.TtlServiceTicket,
              .Threads = Runtime_.Config().Pregen.Threads,
          }))
        , DiskCache_(Runtime_.Config().Pregen.RawListFile,
                     Runtime_.Config().Pregen.KeyTtl,
                     Runtime_.Config().Pregen.TicketMinTtl)
        , AllKeys_(DiskCache_.ExtractKeys())
        , CurrentStorage_(std::make_shared<TStorage::TData>(DiskCache_.ExtractTickets()))
        , NewbiesStorage_(*Generator_)
        , UnistatTime_("ticket_pregeneration.duration", CreateBounds())
    {
        if (AllKeys_.GetDataForPregeneration().empty()) {
            Run();
        } else {
            TLog::Info() << "Pregeneration: got tickets from disk. Skipping generating on start";
        }
    }

    TFacade::~TFacade() = default;

    TString TFacade::GetTicket(NTvmAuth::TTvmId src,
                               NTvmAuth::TTvmId dst) const {
        const TKey k{src, dst};
        AllKeys_.TryAdd(k);

        ++UnistatAll_;
        const TString res = CurrentStorage_.GetTicket(k);
        if (res) {
            ++UnistatHit_;
            return res;
        }

        const time_t expireTime = time(nullptr) + Runtime_.Config().Gen.TtlServiceTicket;
        const TStorageForNewbies::TTicket t = NewbiesStorage_.GetTicket(
            k,
            *Runtime_.DbFetcher().PrivateKey(),
            *Runtime_.DbFetcher().ServiceCtx(),
            expireTime);

        if (t.IsCache) {
            ++UnistatCache_;
        }

        return t.Body;
    }

    void TFacade::AddUnistat(NUnistat::TBuilder& builder) const {
        Generator_->AddUnistat(builder);
        builder.Add(UnistatHit_);
        builder.Add(UnistatCache_);
        builder.Add(UnistatAll_);
        UnistatTime_.AddUnistat(builder);
    }

    void TFacade::Run() {
        TLog::Info() << "Pregeneration: starting";
        const TInstant start = TInstant::Now();

        const TKeyList::TData& keys = AllKeys_.GetDataForPregeneration();
        const TInstant keysReady = TInstant::Now();
        TStorage::TDataPtr res = Generator_->GenAll(keys,
                                                    Runtime_.DbFetcher().PrivateKey(),
                                                    *Runtime_.DbFetcher().ServiceCtx());

        const TDuration pregenTime = TInstant::Now() - start;
        UnistatTime_.Insert(pregenTime);

        CurrentStorage_.SetData(res);
        NewbiesStorage_.Reset();

        TLog::Info() << "Pregeneration: finished. Took " << pregenTime << " for " << keys.size() << " keys."
                     << " Keys were locked for " << (keysReady - start)
                     << ". Threads: " << Runtime_.Config().Pregen.Threads;

        DiskCache_.WriteCache(keys, *res);
    }
}
