#include "generator.h"

#include <passport/infra/daemons/tvmapi/src/runtime_context/config.h>

#include <passport/infra/libs/cpp/tvm/signer/signer.h>
#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/thread/pool.h>

namespace NPassport::NTvm::NPregen {
    TGenerator::TGenerator(TSettings settings)
        : Settings_(settings)
    {
    }

    TStorage::TDataPtr TGenerator::GenAll(const TKeyList::TData& keys,
                                          TRwPrivateKeyPtr pk,
                                          const NTvmAuth::TServiceContext& serviceCtx) {
        TStorage::TDataPtr res = std::make_shared<TStorage::TData>();
        res->max_load_factor(0.5);
        res->reserve(keys.size());

        const time_t expireTime = time(nullptr) + Settings_.TicketTtl;

        std::mutex m;

        TThreadPool pool;
        pool.Start(Settings_.Threads);

        TKeyList::ProcessData(
            keys,
            Settings_.KeyTtl,
            [&, this](const TKey& k, ui32) {
                pool.SafeAddFunc([this, &res, &m, &serviceCtx, k, expireTime, pk]() {
                    TString ticket = GenOne(k, *pk, serviceCtx, expireTime);
                    std::unique_lock lock(m);
                    res->emplace(k, std::move(ticket));
                });
            });

        pool.Stop();

        return res;
    }

    TString TGenerator::GenOne(const TKey& key,
                               const NTvmAuth::NRw::TRwPrivateKey& pk,
                               const NTvmAuth::TServiceContext& serviceCtx,
                               time_t expireTime) {
        int tries = 42;

        while (tries-- > 0) {
            NTicketSigner::TServiceSigner builder;
            builder.SetSides(key.Src, key.Dst);

            TString res = builder.SerializeV3(pk, expireTime);

            NTvmAuth::ETicketStatus st = serviceCtx.Check(res).GetStatus();
            if (st == NTvmAuth::ETicketStatus::InvalidDst || st == NTvmAuth::ETicketStatus::Ok) {
                return res;
            }

            ++UnistatErrors_;
            TLog::Error() << "Failed to generate service ticket: src=" << key.Src
                          << ", dst=" << key.Dst
                          << ", keyid=" << pk.GetId()
                          << ", body=" << res
                          << ". Status: " << NTvmAuth::StatusToString(st);
        }

        ythrow yexception() << "Failed to generate service ticket: totally unexpected scenario. "
                            << "MAYBE key " << pk.GetId() << " is bad";
    }

    void TGenerator::AddUnistat(NUnistat::TBuilder& builder) const {
        builder.Add(UnistatErrors_);
    }
}
