#include "storage.h"

#include "generator.h"
#include "keys.h"

#include <passport/infra/daemons/tvmapi/src/utils/utils.h>

#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/generic/string.h>
#include <util/stream/format.h>

#include <thread>

namespace NPassport::NTvm::NPregen {
    TStorage::TStorage(TDataPtr data) {
        Data_.Set(std::move(data));
    }

    void TStorage::SetData(TStorage::TDataPtr data) {
        Data_.Set(std::move(data));
    }

    TString TStorage::GetTicket(const TKey& k) const {
        const TDataPtr data = Data_.Get();

        auto it = data->find(k);
        if (it == data->end()) {
            return {};
        }

        return it->second;
    }

    TStorageForNewbies::TStorageForNewbies(TGenerator& generator)
        : Data_(std::make_shared<TStorage::TData>())
        , Generator_(generator)
    {
        Reset();
    }

    void TStorageForNewbies::Reset() {
        PrevData_ = std::atomic_load(&Data_);
        std::atomic_store(&Data_, std::make_shared<TStorage::TData>());
    }

    TStorageForNewbies::TTicket TStorageForNewbies::GetTicket(const TKey& k,
                                                              const NTvmAuth::NRw::TRwPrivateKey& pk,
                                                              const NTvmAuth::TServiceContext& serviceCtx,
                                                              time_t expireTime) {
        TStorage::TDataPtr currData = std::atomic_load(&Data_);
        {
            // Already exists in current
            std::shared_lock lock(Mutex_);
            auto it = currData->find(k);
            if (it != currData->end()) {
                return {it->second, true};
            }

            // Already exists in previous
            TStorage::TDataPtr prevData = std::atomic_load(&PrevData_);
            it = prevData->find(k);
            if (it != prevData->end()) {
                return {it->second, true};
            }
        }

        TLog::Info() << "Pregeneration: storage miss and cache miss. src=" << k.Src << ". dst=" << k.Dst;

        // Need to gen
        TString t = Generator_.GenOne(k, pk, serviceCtx, expireTime);
        std::unique_lock lock(Mutex_);
        currData->insert({k, t});
        return {t, false};
    }
}
