#include "request_context.h"

#include <passport/infra/daemons/tvmapi/src/exception.h>
#include <passport/infra/daemons/tvmapi/src/pregeneration/facade.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/experiment.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/notify_log.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/runtime_context.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/strings.h>
#include <passport/infra/daemons/tvmapi/src/utils/retry_settings.h>
#include <passport/infra/daemons/tvmapi/src/utils/utils.h>

#include <passport/infra/libs/cpp/json/writer.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/tvm/signer/signer.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NTvm {
    TRequestContext::TRequestContext(const NCommon::TRequest& req, const TRuntimeContext& runtime)
        : Req_(req)
        , Runtime_(runtime)
    {
    }

    bool TRequestContext::CheckTs() {
        const auto intTs = TUtils::GetRequiredNumArg<time_t>(Req_, TStrings::TS_);

        if (TUtils::CheckTs(intTs, Runtime_.Config().Gen.TsAllowedDiff)) {
            return true;
        }

        TLog::Debug() << "Request failed. Incorrect 'ts'";
        TString msg = NUtils::CreateStr(
            "Incorrect 'ts': ", intTs,
            " (", ToString(TInstant::Seconds(intTs)), "). ",
            "Time differs by more than a minute: between request (your instance time) and tvm-api.yandex.net time."
            " Please, fix your NTP: https://wiki.yandex-team.ru/passport/tvm2/400/");

        Res_ = TResult(std::move(msg), TStatus(TStatus::STR_ERR_REQUEST, TStatus::STR_INCORRECT_TS));
        return false;
    }

    static const TString MSG_INIT = "Failed to init";
    bool TRequestContext::GetClient() {
        const TString& src = TUtils::GetRequiredUIntArg(Req_, TStrings::SRC_);
        ClientCtx_ = Runtime_.DbFetcher().GetClient(IntFromString<TClientId, 10>(src));

        if (!ClientCtx_) {
            TLog::Debug() << "Request failed. Src is not found: " << src;
            Res_ = TResult("Src is not found: it never existed. Or "
                           "it was created a moment ago and tvm-api needs couple minutes to get it. "
                           "client_id=" +
                               src,
                           TStatus::ERROR_CLIENT__NOT_FOUND);
            return false;
        }

        if (ClientCtx_.Client().IsDeleted()) {
            Runtime_.NotifyLog().Log(TNotifyLog::TDeletedSrc(ClientCtx_, Req_.GetRemoteAddr()));
        }

        if (!ClientCtx_.Client().IsCorrect()) {
            TLog::Debug() << "v2: Client is incorrect: " << src;
            Res_ = TResult("Src is incorrect: secret is missing. client_id=" + src,
                           TStatus::ERROR_CLIENT__INCORRECT);
            return false;
        }

        if (!ClientCtx_.IsKeyOk()) {
            TLog::Debug() << "Request failed. " << MSG_INIT;
            Res_ = TResult(MSG_INIT, TStatus::SERVICE_UNAVAILABLE);
            return false;
        }

        return true;
    }

    bool TRequestContext::CheckServiceTicket() {
        const TString& serviceTicket = Req_.GetHeader(TStrings::X_YA_SERVICE_TICKET_);
        if (serviceTicket.empty()) {
            TLog::Debug() << "Missing service-ticket";
            Res_ = TResult("ServiceTicket is required", TStatus::ERROR_CLIENT__NOT_FOUND);
            return false;
        }

        NTvmAuth::TCheckedServiceTicket t = Runtime_.DbFetcher().ServiceCtx()->Check(serviceTicket);
        if (!t) {
            TLog::Debug() << "Failed to check service ticket: " << NTvmAuth::StatusToString(t.GetStatus())
                          << ". Body: " << NTvmAuth::NUtils::RemoveTicketSignature(serviceTicket);
            Res_ = TResult("ServiceTicket is incorrect: " + (TString)NTvmAuth::StatusToString(t.GetStatus()),
                           TStatus::ERROR_CLIENT__INCORRECT);
            return false;
        }

        ClientCtx_ = Runtime_.DbFetcher().GetClient(t.GetSrc());
        if (!ClientCtx_) {
            TLog::Debug() << "Request failed. Src is not found";
            Res_ = TResult("Src is not found", TStatus::ERROR_CLIENT__NOT_FOUND);
            return false;
        }

        return true;
    }

    TResult TRequestContext::Result() {
        Runtime_.GetRetrySettingsHelper().Process(Req_, Res_);
        return std::move(Res_);
    }

    TDbFetcher::TResult TRequestContext::Client() {
        return ClientCtx_;
    }
}
