#include "response_builder.h"

#include <passport/infra/daemons/tvmapi/src/exception.h>
#include <passport/infra/daemons/tvmapi/src/output/result.h>
#include <passport/infra/daemons/tvmapi/src/pregeneration/facade.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/notify_log.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/runtime_context.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/strings.h>

#include <passport/infra/libs/cpp/json/writer.h>
#include <passport/infra/libs/cpp/tvm/signer/signer.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/generic/strbuf.h>
#include <util/stream/str.h>

namespace NPassport::NTvm {
    TResponseBuilder::TResponseBuilder(const TRuntimeContext& runtime)
        : Runtime_(runtime)
    {
    }

    TResult TResponseBuilder::BuildOkResult(const TStringBuf dst,
                                            const TDbFetcher::TResult& client,
                                            const TDbFetcher::TResult& srcRes,
                                            const TString& remoteAddr) {
        return TResult(BuildOkJson(dst, client, srcRes, remoteAddr),
                       TResult::EContentType::TextJson);
    }

    static const TString ERROR_ = "error";
    static const TString TICKET_ = "ticket";
    TString TResponseBuilder::BuildOkJson(const TStringBuf dst,
                                          const TDbFetcher::TResult& client,
                                          const TDbFetcher::TResult& srcRes,
                                          const TString& remoteAddr) {
        const time_t expireTime = time(nullptr) + Runtime_.Config().Gen.TtlServiceTicket;

        TString buf;
        {
            NJson::TWriter wr(buf);
            NJson::TObject root(wr);

            std::vector<TStringBuf> dsts = NUtils::NormalizeListValue<TStringBuf>(dst, ",");

            const size_t dstLimit = Runtime_.Config().Gen.DstCountLimit;
            if (dsts.size() > Runtime_.Config().Gen.DstCountWarningLimit) {
                TLog::Warning() << "Too many dsts in request: " << dsts.size()
                                << " (limit=" << dstLimit
                                << "). Src=" << client.Client().Id();
            }
            if (dsts.size() > dstLimit) {
                throw TIncorrectArgException(TStrings::DST_)
                    << "Arg 'dst' contains " << dsts.size() << " client_ids. Allowed limit=" << dstLimit;
            }

            for (const TStringBuf currDst : dsts) {
                TClientId dstId = 0;
                if (!TryIntFromString<10>(currDst, dstId)) {
                    throw TIncorrectArgException(TStrings::DST_) << "Arg 'dst' must be comma-separated array of numbers";
                }
                NJson::TObject obj(root, currDst);

                TDbFetcher::TResult dstRes = Runtime_.DbFetcher().GetClient(dstId);
                if (!dstRes) {
                    TLog::Warning() << "Dst was not found: " << dstId << ". src=" << client.Client().Id();
                    TStringStream s;
                    s << "Dst is not found: it never existed. Or "
                         "it was created a moment ago and tvm-api needs couple minutes to get it. "
                         "client_id="
                      << currDst;
                    obj.Add(ERROR_, s.Str());
                } else {
                    WriteTicket(obj, client, dstId, expireTime);
                }

                if (dstRes && dstRes.Client().IsDeleted()) {
                    Runtime_.NotifyLog().Log(TNotifyLog::TDeletedDst(
                        srcRes,
                        remoteAddr,
                        dstRes));
                }
            }
        }

        return buf;
    }

    TClientBuilder::TClientBuilder(const TRuntimeContext& runtime)
        : TResponseBuilder(runtime)
    {
    }

    void TClientBuilder::WriteTicket(NJson::TObject& obj,
                                     const TDbFetcher::TResult& client,
                                     TClientId dst,
                                     time_t expireTime) {
        if (Runtime_.Config().Experiment.UsePregeneration) {
            TString t = Runtime_.Pregeneration().GetTicket(client.Client().Id(), dst);
            if (t) {
                obj.Add(TICKET_, t);
                return;
            }

            TLog::Info() << "Perfomance issue: pregeneration failed for grant_type=client_credendials";
        }

        // Straws
        NTicketSigner::TServiceSigner builder;
        builder.SetSides(client.Client().Id(), dst);

        obj.Add(TICKET_, builder.SerializeV3(client.Key(), expireTime));
    }

    TSshBuilder::TSshBuilder(const TRuntimeContext& runtime, ui64 uid)
        : TResponseBuilder(runtime)
        , Uid_(uid)
    {
        Y_ENSURE(Uid_ > 0);
    }

    void TSshBuilder::WriteTicket(NJson::TObject& obj,
                                  const TDbFetcher::TResult& client,
                                  TClientId dst,
                                  time_t expireTime) {
        NTicketSigner::TServiceSigner builder;
        builder.SetSides(client.Client().Id(), dst);
        builder.SetIssuerUid(Uid_);

        obj.Add(TICKET_, builder.SerializeV3(client.Key(), expireTime));
    }

    TString TVerifySshBuilder::BuildOk(TStringBuf fingerprint) {
        TString buf;

        NJson::TWriter wr(buf);
        NJson::TObject root(wr);
        root.Add("status", "OK");
        root.Add("fingerprint", fingerprint);

        return buf;
    }
}
