#include "check_secret.h"

#include <passport/infra/daemons/tvmapi/src/processors/request_context.h>

#include <passport/infra/daemons/tvmapi/src/output/result.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/runtime_context.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/strings.h>

#include <passport/infra/libs/cpp/json/writer.h>
#include <passport/infra/libs/cpp/request/request.h>

namespace NPassport::NTvm::NV2 {
    TCheckSecretProcessor::TCheckSecretProcessor(const TRuntimeContext& runtime)
        : Runtime_(runtime)
    {
    }

    static const TString CLIENT_IS_NOT_ALLOWED = "Client is not allowed for verify_ssh";

    TResult TCheckSecretProcessor::Process(const NCommon::TRequest& req) const {
        const TString& secretHeader = req.GetHeader(TStrings::X_YA_SECRET_);
        if (secretHeader.empty()) {
            return TResult("Header '" + TStrings::X_YA_SECRET_ + "' is required", TStatus::ERROR_REQUEST__MISSING);
        }
        const TString secret = NUtils::Base64url2bin(secretHeader);
        if (secret.empty()) {
            return TResult("Secret is malformed", TStatus::ERROR_REQUEST__MISSING);
        }

        TRequestContext ctx(req, Runtime_);
        if (!ctx.CheckServiceTicket()) {
            return ctx.Result();
        }

        TDbFetcher::TResult clientCtx = ctx.Client();
        if (!Runtime_.IsCheckSecretClientIdAllowed(clientCtx.Client().Id())) {
            TLog::Debug() << CLIENT_IS_NOT_ALLOWED << ": " << clientCtx.Client().Id();
            return TResult(CLIENT_IS_NOT_ALLOWED, TStatus::ERROR_CLIENT__INCORRECT);
        }

        TSecretIndexPtr idx = Runtime_.DbFetcher().SecretIndex();
        Y_ENSURE(idx);

        auto pair = idx->equal_range(secret);
        if (pair.first == pair.second) {
            return TResult("", TStatus::REGULAR_NOT_FOUND);
        }

        TString res;
        {
            NJson::TWriter wr(res);
            NJson::TArray arr(wr);
            for (auto it = pair.first; it != pair.second; ++it) {
                const TClient& cl = *it->second;

                NJson::TObject obj(arr);
                obj.Add("client_id", cl.Id());

                std::optional<ui64> abcId = cl.GetAbcId();
                if (abcId) {
                    obj.Add("abc_id", *abcId);
                } else {
                    obj.Add("abc_id", nullptr);
                }

                obj.Add("primary", it->first == cl.Secret());
            }
        }

        return TResult(std::move(res), TResult::EContentType::TextJson);
    }
}
