#include "client_credentials.h"

#include <passport/infra/daemons/tvmapi/src/processors/request_context.h>
#include <passport/infra/daemons/tvmapi/src/processors/response_builder.h>

#include <passport/infra/daemons/tvmapi/src/output/result.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/experiment.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/notify_log.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/runtime_context.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/strings.h>
#include <passport/infra/daemons/tvmapi/src/utils/retry_settings.h>
#include <passport/infra/daemons/tvmapi/src/utils/utils.h>

#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/unistat/consumers.h>

#include <library/cpp/tvmauth/src/utils.h>

#include <util/stream/str.h>
#include <util/string/cast.h>

namespace NPassport::NTvm::NV2 {
    TClCredProcessor::TClCredProcessor(const TRuntimeContext& runtime,
                                       NUnistat::TSimpleConsumers& consumersWithOld)
        : Runtime_(runtime)
        , ConsumersWithOld_(consumersWithOld)
    {
    }

    static const TString ERR_MISSING = "Missing required param: deprecated_env_sign OR deprecated_ts_sign";
    TResult TClCredProcessor::Process(const NCommon::TRequest& req) const {
        TRequestContext ctx(req, Runtime_);
        if (!ctx.CheckTs() || !ctx.GetClient()) {
            return ctx.Result();
        }

        const TString& dst = TUtils::GetRequiredArg(req, TStrings::DST_);
        TUtils::CheckArgAsNumList<uint32_t>(dst, TStrings::DST_);
        // ts was already checked in CheckTs, function below throws
        const TString ts = ToString(TUtils::GetRequiredNumArg<time_t>(req, TStrings::TS_));

        const TDbFetcher::TResult client = ctx.Client();
        Y_ENSURE(client);
        const TString sign = NUtils::Base64url2bin(TUtils::GetRequiredArg(req, TStrings::SIGN_));
        const TString& scopes = req.GetArg(TStrings::SCOPES_);
        TStringStream toSign;
        toSign.Reserve(ts.size() + dst.size() + scopes.size() + 3);
        toSign << ts << "|" << dst << "|" << scopes << "|";

        const TClient::ECheckStatus status = client.Client().CheckSecret(sign, toSign.Str());
        if (TClient::ECheckStatus::Invalid == status) {
            TLog::Debug() << "Request failed: Sign don't match. client_id=" << client.Client().Id();
            TResult res("Signature is bad: common reason is bad tvm_secret or tvm_id/tvm_secret mismatch. https://wiki.yandex-team.ru/passport/tvm2/400/",
                        TStatus::ERROR_REQUEST__BROKEN_SIGN);
            Runtime_.GetRetrySettingsHelper().Process(req, res);
            return res;
        }

        if (TClient::ECheckStatus::ValidWithOldSecret == status) {
            ConsumersWithOld_.Add(TUtils::GetRequiredUIntArg(req, TStrings::SRC_));
            Runtime_.NotifyLog().Log(TNotifyLog::TOldSecret(client, req.GetRemoteAddr()));
        }

        TClientBuilder b(Runtime_);
        TResult res = b.BuildOkResult(dst, client, ctx.Client(), req.GetRemoteAddr());
        Runtime_.GetRetrySettingsHelper().Process(req, res);
        return res;
    }
}
