#include "sshkey.h"

#include <passport/infra/daemons/tvmapi/src/processors/request_context.h>
#include <passport/infra/daemons/tvmapi/src/processors/response_builder.h>

#include <passport/infra/daemons/tvmapi/src/exception.h>
#include <passport/infra/daemons/tvmapi/src/output/result.h>
#include <passport/infra/daemons/tvmapi/src/output/status.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/abc_fetcher.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/dbfetcher.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/runtime_context.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/staff_fetcher.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/strings.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/tvmcert_client.h>
#include <passport/infra/daemons/tvmapi/src/utils/client.h>
#include <passport/infra/daemons/tvmapi/src/utils/utils.h>

#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NTvm::NV2 {
    TSshkeyProcessor::TSshkeyProcessor(const TRuntimeContext& runtime)
        : Runtime_(runtime)
    {
    }

    static const TString CLIENT_IS_NOT_ALLOWED = "Client is not allowed for verify_ssh";

    static TString ErrorMessage(TStringBuf err) {
        return NUtils::CreateStr("Ssh sign is broken: ", err);
    }

    TResult TSshkeyProcessor::Process(const NCommon::TRequest& req) const {
        const TString& src = ToString(TUtils::GetRequiredNumArg<uint32_t>(req, TStrings::SRC_));
        const TString& dst = TUtils::GetRequiredArg(req, TStrings::DST_);
        TUtils::CheckArgAsNumList<uint32_t>(dst, TStrings::DST_);
        const TString& sshSign = TUtils::GetRequiredArg(req, TStrings::SSH_SIGN_);
        const TString& publicCert = req.GetArg(TStrings::SSH_PUBLIC_CERT_);
        const TString& uid = TUtils::GetUIntArg(req, TStrings::UID_);
        const TString login = NUtils::TrimCopy(req.GetArg(TStrings::LOGIN_));

        if ((uid.empty() && login.empty()) || (!uid.empty() && !login.empty())) {
            TLog::Debug() << "SshKey: login not found: 'login'('" << login
                          << "') OR 'uid' ('" << uid << "') ";
            static const TResult RES("Required 'login' OR 'uid'",
                                     TStatus::ERROR_REQUEST__MISSING);
            return RES;
        }

        TRequestContext ctx(req, Runtime_);
        if (!ctx.CheckTs() || !ctx.GetClient()) {
            return ctx.Result();
        }

        ui64 uidInt = uid.empty() ? Runtime_.StaffFetcher().LoginToUid(login)
                                  : IntFromString<ui64, 10>(uid);
        if (0 == uidInt) {
            static const TString msg = "User not found";
            TLog::Debug() << "Request failed. " << msg;
            return TResult(msg, TStatus::ERROR_CRED__NO_SSH_KEY);
        }

        TDbFetcher::TResult client = ctx.Client();
        std::optional<ui64> abcId = client.Client().GetAbcId();
        if (!abcId || !Runtime_.AbcFetcher().CheckRole(uidInt, *abcId)) {
            TLog::Debug() << "Request failed. User does not have grant for client";

            return TResult(
                abcId
                    ? NUtils::CreateStr(
                          "User ", uidInt,
                          " does not have role 'TVM ssh user' for tvm_id=", client.Client().Id(),
                          ": get it in ABC service https://abc.yandex-team.ru/services/", *abcId,
                          "/. Use this guide, please: https://wiki.yandex-team.ru/passport/tvm2/getabcrole/")
                    : NUtils::CreateStr(
                          "User ", uidInt,
                          " does not have role 'TVM ssh user' for tvm_id=", client.Client().Id(),
                          ": nobody can get it because tvm_id is not linked with any ABC service",
                          ". You can link it with email to tvm-dev@yandex-team.ru"),
                TStatus::ERROR_CRED__UID_NOT_ALLOWED);
        }

        TString loginStaff = login.empty() ? Runtime_.StaffFetcher().UidToLogin(uidInt) : login;

        // ts was already checked in CheckTs, function below throws
        const time_t ts = TUtils::GetRequiredNumArg<time_t>(req, TStrings::TS_);
        const TString sshString = NUtils::CreateStr(ts, "|", src, "|", dst);

        TString err;
        if (publicCert.empty()) {
            TStaffFetcher::TCheckResult result = Runtime_.StaffFetcher().CheckSign(
                uidInt,
                NUtils::Base64url2bin(sshSign),
                sshString);

            err = std::move(result.Err);
        } else {
            TTvmCertClient::TCheckResult result = Runtime_.TvmCertClient().CheckSign(
                req.GetRequestId(),
                loginStaff,
                publicCert,
                sshSign,
                sshString);

            err = std::move(result.Err);
        }

        if (err) {
            const TString msg = ErrorMessage(err);
            TLog::Debug() << "Request failed. grant_type=sshkey. " << msg
                          << ". uid=" << uidInt << " (" << loginStaff << ")";
            return TResult(msg, TStatus::ERROR_CRED__SSH_BROKEN);
        }

        TSshBuilder b(Runtime_, uidInt);
        return b.BuildOkResult(dst, client, ctx.Client(), req.GetRemoteAddr());
    }

    TResult TSshkeyProcessor::VerifySsh(const NCommon::TRequest& req) const {
        const TString& sshSign = TUtils::GetRequiredArg(req, TStrings::SSH_SIGN_);
        const TString& uid = TUtils::GetUIntArg(req, TStrings::UID_);
        const TString login = NUtils::TrimCopy(req.GetArg(TStrings::LOGIN_));
        const TString& toSign = TUtils::GetRequiredArg(req, TStrings::TO_SIGN_);
        const TString& publicCert = req.GetArg(TStrings::SSH_PUBLIC_CERT_);

        if ((uid.empty() && login.empty()) || (!uid.empty() && !login.empty())) {
            TLog::Debug() << "VerifySsh: required 'login'('" << login
                          << "') OR 'uid' ('" << uid << "') ";
            return TResult("Required 'login' OR 'uid'", TStatus::ERROR_REQUEST__MISSING);
        }

        TRequestContext ctx(req, Runtime_);
        if (!ctx.CheckServiceTicket()) {
            return ctx.Result();
        }

        TDbFetcher::TResult clientCtx = ctx.Client();
        if (!Runtime_.IsVerifierSshClientIdAllowed(clientCtx.Client().Id())) {
            TLog::Debug() << CLIENT_IS_NOT_ALLOWED << ": " << clientCtx.Client().Id();
            return TResult(CLIENT_IS_NOT_ALLOWED, TStatus::ERROR_CLIENT__INCORRECT);
        }

        ui64 uidInt = uid.empty() ? Runtime_.StaffFetcher().LoginToUid(login)
                                  : IntFromString<ui64, 10>(uid);
        if (0 == uidInt) {
            TLog::Debug() << "VerifySsh: login not found: 'login'('" << login
                          << "') OR 'uid' ('" << uid << "') ";
            return TResult("Login not found", TStatus::ERROR_CRED__NO_SSH_KEY);
        }

        TString loginStaff = login.empty() ? Runtime_.StaffFetcher().UidToLogin(uidInt) : login;

        TString err;
        TString fingerprint;
        if (publicCert.empty()) {
            TStaffFetcher::TCheckResult result = Runtime_.StaffFetcher().CheckSign(
                uidInt,
                NUtils::Base64url2bin(sshSign),
                toSign);

            err = std::move(result.Err);
            fingerprint = std::move(result.Fingerprint);
        } else {
            TTvmCertClient::TCheckResult result = Runtime_.TvmCertClient().CheckSign(
                req.GetRequestId(),
                loginStaff,
                publicCert,
                sshSign,
                toSign);

            err = std::move(result.Err);
        }

        if (err) {
            const TString msg = ErrorMessage(err);
            TLog::Debug() << "Request failed. verify_ssh. " << msg
                          << ". uid=" << uidInt << " (" << login << ")";
            return TResult(msg, TStatus::ERROR_CRED__SSH_BROKEN);
        }

        return TResult(TVerifySshBuilder::BuildOk(fingerprint),
                       TResult::EContentType::TextJson);
    }
}
