#include "abc_fetcher.h"

#include "runtime_context.h"

#include <passport/infra/daemons/tvmapi/src/utils/utils.h>

#include <passport/infra/libs/cpp/dbpool/result.h>
#include <passport/infra/libs/cpp/dbpool/util.h>
#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/http/simple/http_client.h>

#include <util/generic/yexception.h>
#include <util/string/cast.h>

#include <atomic>

namespace NPassport::NTvm {
    static const TString TVM_SSH_USER = "tvm_ssh_user";

    TAbcFetcher::TAbcFetcher(const TRuntimeContext& runtime)
        : Runtime_(runtime)
        , UnistatResponseTime_("responsetime.abc", NUnistat::TTimeStat::CreateBoundsFromMaxValue(Runtime_.Config().Abc.QueryTimeout))
    {
        Y_ENSURE(Runtime_.Config().Staff.Enabled);

        if (!Runtime_.Config().Abc.CachePath.empty()) {
            try {
                ReadCacheFromFile();
                TLog::Info() << "AbcFetcher: loaded from file";
            } catch (const std::exception& e) {
                TLog::Error() << "AbcFetcher: failed to get info from file: " << e.what();
            }
        }

        if (!Cache_.Get()) {
            try {
                RefreshViaHttp();
                TLog::Info() << "AbcFetcher: loaded from HTTP";
            } catch (const std::exception& e) {
                TLog::Error() << "AbcFetcher: failed to get info from Abc: " << e.what();
            }
        }

        if (!Cache_.Get()) {
            Cache_.Set(std::make_shared<TCache>());
        }
    }

    TAbcFetcher::~TAbcFetcher() = default;

    void TAbcFetcher::AddUnistat(NUnistat::TBuilder& builder) const {
        builder.Add(UnistatQueryErrors_);
        builder.Add(UnistatParsingErrors_);
        UnistatResponseTime_.AddUnistat(builder);
    }

    bool TAbcFetcher::CheckRole(ui64 uid, ui64 serviceId) const {
        const TCachePtr cache = Cache_.Get();

        auto it = cache->Users.find(uid);
        if (it == cache->Users.end()) {
            return false;
        }

        return std::find(it->second.begin(), it->second.end(), serviceId) != it->second.end();
    }

    void TAbcFetcher::Run() {
        RefreshViaHttp();
    }

    void TAbcFetcher::ReadCacheFromFile() {
        rapidjson::Document doc;
        Y_ENSURE(NJson::TReader::DocumentAsArray(NUtils::ReadFile(Runtime_.Config().Abc.CachePath), doc));

        TCachePtr cache = std::make_shared<TCache>();
        cache->Users.reserve(1000);
        size_t totalRoles = 0;
        for (rapidjson::SizeType idx = 0; idx < doc.Size(); ++idx) {
            try {
                totalRoles += ParsePage(doc[idx], *cache).RoleCount;
            } catch (const std::exception& e) {
                TLog::Error() << "Exception on parsing ABC cache: " << e.what();
                throw;
            }
        }
        TLog::Info() << "AbcFetcher: " << cache->Users.size()
                     << " logins with " << totalRoles << " roles"
                     << ". From: " << Runtime_.Config().Abc.CachePath;

        Cache_.Set(std::move(cache));
    }

    void TAbcFetcher::RefreshViaHttp() {
        TLog::Info() << "AbcFetcher: start refresh from abc";
        if (Token_.empty()) {
            Token_ = "OAuth " + Runtime_.GetOAuthToken();
        }

        TKeepAliveHttpClient client(Runtime_.Config().Abc.Host,
                                    Runtime_.Config().Abc.Port,
                                    Runtime_.Config().Abc.QueryTimeout,
                                    Runtime_.Config().Abc.ConnectionTimeout);

        const size_t expectedLoginCount = 100000;
        std::vector<TString> pages;
        pages.reserve(expectedLoginCount / Runtime_.Config().Abc.Limit + 1);

        TCachePtr cache = std::make_shared<TCache>();
        cache->Users.reserve(expectedLoginCount);

        size_t totalRoles = 0;
        size_t lastEntityId = 0;

        while (true) {
            TResp resp = GetPage(client, lastEntityId);

            TErrorIncrementer err{&UnistatParsingErrors_};

            rapidjson::Document doc;
            if (!NJson::TReader::DocumentAsObject(resp.Body, doc)) {
                throw yexception() << "http response from ABC is not object. " << resp.Body;
            }

            try {
                const TAbcFetcher::TParsePageResult result = ParsePage(doc, *cache);

                if (result.RoleCount == 0) {
                    TLog::Debug() << "AbcFetcher: Got empty list of roles from ABC"
                                  << ". Took: " << resp.Time
                                  << ". Stop fetching";
                    err.Err = nullptr;
                    break;
                }
                TLog::Debug() << "AbcFetcher: Got " << result.RoleCount << " roles from ABC"
                              << ". Took: " << resp.Time
                              << ". Continue fetching";

                totalRoles += result.RoleCount;
                lastEntityId = result.LastId;
            } catch (const std::exception& e) {
                TLog::Error() << "Exception on parsing ABC response: " << e.what() << ".body: " << resp.Body;
                throw;
            }

            pages.push_back(std::move(resp.Body));
            err.Err = nullptr;
        }

        TLog::Info() << "AbcFetcher: " << cache->Users.size() << " logins with " << totalRoles << " roles";
        Cache_.Set(std::move(cache));

        TUtils::WriteJsonArrayToFile(pages, Runtime_.Config().Abc.CachePath);
    }

    TAbcFetcher::TResp TAbcFetcher::GetPage(TKeepAliveHttpClient& client, size_t entityId) const {
        TString url = NUtils::CreateStrExt(
            32,
            "/api/v4/services/members/",
            "?format=json",
            "&fields=role.code,person.uid,service.id,id",
            "&ordering=id",
            "&role__code=",
            TVM_SSH_USER,
            "&page_size=",
            Runtime_.Config().Abc.Limit);
        if (entityId > 0) {
            NUtils::Append(url, "&id__gt=", entityId);
        }

        TString output;
        size_t retries = 0;
        TDuration respTime;
        bool success = TUtils::FetchWithRetries(
            client,
            url,
            UnistatResponseTime_,
            Runtime_.Config().Abc.Retries,
            Token_,
            output,
            retries,
            respTime);
        UnistatQueryErrors_ += retries;
        Y_ENSURE(success, "AbcFetcher: Failed to fetch roles from Abc");

        return {output, respTime};
    }

    TAbcFetcher::TParsePageResult TAbcFetcher::ParsePage(rapidjson::Value& doc, TAbcFetcher::TCache& cache) {
        const rapidjson::Value* jResultArray = nullptr;
        Y_ENSURE(NJson::TReader::MemberAsArray(doc, "results", jResultArray));

        TAbcFetcher::TParsePageResult res;
        for (std::size_t resIdx = 0; resIdx < jResultArray->Size(); ++resIdx) {
            const rapidjson::Value& jItem = (*jResultArray)[resIdx];

            ui64 id = 0;
            Y_ENSURE(NJson::TReader::MemberAsUInt64(jItem, "id", id));
            res.LastId = std::max(res.LastId, id);

            const rapidjson::Value* jRole = nullptr;
            Y_ENSURE(NJson::TReader::MemberAsObject(jItem, "role", jRole));
            TString role;
            Y_ENSURE(NJson::TReader::MemberAsString(*jRole, "code", role));
            if (role != TVM_SSH_USER) {
                continue;
            }

            const rapidjson::Value* jPerson = nullptr;
            Y_ENSURE(NJson::TReader::MemberAsObject(jItem, "person", jPerson));
            TString uidStr;
            Y_ENSURE(NJson::TReader::MemberAsString(*jPerson, "uid", uidStr));

            const rapidjson::Value* jService = nullptr;
            Y_ENSURE(NJson::TReader::MemberAsObject(jItem, "service", jService));
            ui64 serviceId = 0;
            Y_ENSURE(NJson::TReader::MemberAsUInt64(*jService, "id", serviceId));

            ui64 uid = IntFromString<ui64, 10>(uidStr);
            auto it = cache.Users.insert({uid, TCache::TServices({})}).first;
            auto itVec = std::find(it->second.begin(), it->second.end(), serviceId);
            if (itVec == it->second.end()) {
                it->second.push_back(serviceId);
                ++res.RoleCount;
            }
        }

        return res;
    }
}
