#include "config.h"

#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>
#include <passport/infra/libs/cpp/xml/config.h>

namespace NPassport::NTvm {
    TConfig::TConfig(const NXml::TConfig& c, const TString& path)
        : Oauth(ReadOAuth(c, path))
        , Staff(ReadStaff(c, path))
        , Abc(ReadAbc(c, path))
        , Experiment(ReadExperiment(c, path))
        , PassportIds(ReadPassportIds(c, path))
        , DbFetcher(ReadDbFetcher(c, path))
        , Pregen(ReadPregeneration(c, path))
        , Gen(ReadGeneral(c, path))
        , RetrySettings(ReadRetrySettings(c, path))
        , TvmCertClientSettings(ReadTvmCertClient(c, path))
    {
    }

    TConfig::TOAuth TConfig::ReadOAuth(const NXml::TConfig& c, const TString& path) {
        TOAuth res;
        res.Schema = c.AsString(path + "/staff/oauth/schema", "http");
        res.Host = c.AsString(path + "/staff/oauth/host");

        const auto readFromFile = [&](const TString& subpath) {
            return NUtils::ReadFile(c.AsString(path + subpath), "");
        };

        res.Login = readFromFile("/staff/oauth/robot_login_file");
        res.Pwd = readFromFile("/staff/oauth/robot_pwd_file");
        res.ClientId = readFromFile("/staff/oauth/client_id_file");
        res.ClientSecret = readFromFile("/staff/oauth/client_secret_file");
        return res;
    }

    TConfig::TStaff TConfig::ReadStaff(const NXml::TConfig& c, const TString& path) {
        TStaff res;
        res.Enabled = c.AsBool(path + "/staff_processor/enabled", false);
        res.RefreshPeriod = c.AsInt(path + "/staff_processor/refresh_period", 3600);
        res.Retries = c.AsInt(path + "/staff_processor/retries_per_request", 3);
        res.CachePath = c.AsString(path + "/staff/cache_path", "");
        res.Limit = c.AsInt(path + "/staff_processor/limit_per_request", 1000);
        Y_ENSURE(res.Limit > 0);
        res.Host = c.AsString(path + "/staff/db_host");
        if (!res.Host.StartsWith("http")) {
            res.Host = "http://" + res.Host;
        }
        res.Port = c.AsNum<ui16>(path + "/staff/db_port", 80);
        res.ConnectionTimeout = TDuration::MilliSeconds(c.AsNum<ui32>(path + "/staff/connect_timeout", 5000));
        res.QueryTimeout = TDuration::MilliSeconds(c.AsNum<ui32>(path + "/staff/query_timeout", 5000));
        return res;
    }

    TConfig::TAbc TConfig::ReadAbc(const NXml::TConfig& c, const TString& path) {
        TAbc res;
        res.Enabled = c.AsBool(path + "/abc/enabled", true);
        res.RefreshPeriod = c.AsInt(path + "/abc/refresh_period", 3600);
        res.Retries = c.AsInt(path + "/abc/retries_per_request", 3);
        res.CachePath = c.AsString(path + "/abc/cache_path", "");
        res.Limit = c.AsInt(path + "/abc/limit_per_request", 1000);
        Y_ENSURE(res.Limit > 0);
        res.Host = c.AsString(path + "/abc/db_host");
        if (!res.Host.StartsWith("http")) {
            res.Host = "http://" + res.Host;
        }
        res.Port = c.AsNum<ui16>(path + "/abc/db_port", 443);
        res.ConnectionTimeout = TDuration::MilliSeconds(c.AsNum<ui32>(path + "/abc/connect_timeout", 5000));
        res.QueryTimeout = TDuration::MilliSeconds(c.AsNum<ui32>(path + "/abc/query_timeout", 5000));
        return res;
    }

    TConfig::TExperiment TConfig::ReadExperiment(const NXml::TConfig& c, const TString& path) {
        TExperiment res;
        res.UsePregeneration = c.AsBool(path + "/experiment/enable_pregeneration", true);
        return res;
    }

    TConfig::TPassportIds TConfig::ReadPassportIds(const NXml::TConfig& c, const TString& path) {
        TPassportIds res;
        res.BbProd = c.AsInt(path + "/passport_ids/bb_prod");
        res.BbProdYateam = c.AsInt(path + "/passport_ids/bb_prod_yateam");
        res.BbTest = c.AsInt(path + "/passport_ids/bb_test");
        res.BbTestYateam = c.AsInt(path + "/passport_ids/bb_test_yateam");
        res.BbStress = c.AsInt(path + "/passport_ids/bb_stress");
        res.BbMimino = c.AsInt(path + "/passport_ids/bb_mimino", 0);
        res.BbLoad = c.AsInt(path + "/passport_ids/bb_load", 0);
        res.Tvm = c.AsInt(path + "/passport_ids/tvm");
        return res;
    }

    TConfig::TDbFetcher TConfig::ReadDbFetcher(const NXml::TConfig& c, const TString& path) {
        TDbFetcher res;
        res.Retries = c.AsInt(path + "/db_fetcher/retries_per_request", 3);
        res.Period = c.AsInt(path + "/db_fetcher/refresh_period", 60);
        res.KeyFile = c.AsString(path + "/db_fetcher/key_file", "/etc/yandex/oauth/secret.key");
        res.PreferedPrivateKeyIdx = c.AsInt(path + "/db_fetcher/prefered_private_key_idx", 6);
        res.DiskCache = c.AsString(path + "/db_fetcher/disk_cache");
        res.MinKeyCount = c.AsNum<size_t>(path + "/db_fetcher/min_key_count", 12);
        return res;
    }

    TConfig::TPregeneration TConfig::ReadPregeneration(const NXml::TConfig& c, const TString& path) {
        TPregeneration res;
        res.Period = c.AsInt(path + "/pregeneration/period", 60);
        res.KeyTtl = c.AsInt(path + "/pregeneration/key_ttl", 86400);
        res.TicketMinTtl = c.AsInt(path + "/pregeneration/ticket_min_ttl", 3600 * 10);
        res.RawListFile = c.AsString(path + "/pregeneration/raw_list_file", "/var/cache/yandex/tvm/raw_list");
        res.Threads = c.AsInt(path + "/pregeneration/threads", 8);
        return res;
    }

    TConfig::TGeneral TConfig::ReadGeneral(const NXml::TConfig& c, const TString& path) {
        TGeneral res;
        res.TsAllowedDiff = c.AsInt(path + "/timestamp_allowed_diff", 600);
        res.ForceDownFile = c.AsString(path + "/force_down_file", "");
        res.VerifySshClientIds = ReadClientIds(c.AsString(path + "/verifyssh_client_id", ""));
        res.CheckSecretClientIds = ReadClientIds(c.AsString(path + "/checksecret_client_id", ""));
        res.TtlServiceTicket = c.AsInt(path + "/ttl_service_ticket", 43200);
        res.KeysAcceptableAge = c.AsNum<time_t>(path + "/keys_acceptable_age");
        res.DstCountLimit = c.AsNum<size_t>(path + "/dst_count_limit", 500);
        res.DstCountWarningLimit = c.AsNum<size_t>(path + "/dst_count_warn_limit", 400);
        return res;
    }

    TConfig::TRetrySettings TConfig::ReadRetrySettings(const NXml::TConfig& c, const TString& path) {
        TRetrySettings res;

        res.ExponentialBackoffMinSec = c.AsNum<ui32>(path + "/retry_settings/exponential_backoff_min_sec", 0);
        res.ExponentialBackoffMaxSec = c.AsNum<ui32>(path + "/retry_settings/exponential_backoff_max_sec", 60);
        res.ExponentialBackoffFactor = c.AsDouble(path + "/retry_settings/exponential_backoff_factor", 2);
        res.ExponentialBackoffJitter = c.AsDouble(path + "/retry_settings/exponential_backoff_jitter", 0.5);
        res.MaxRandomSleepDefaultMsec = c.AsNum<ui32>(path + "/retry_settings/max_random_sleep_default", 5000);
        res.MaxRandomSleepWhenOkMsec = c.AsNum<ui32>(path + "/retry_settings/max_random_sleep_when_ok", 60000);
        res.RetriesOnStart = c.AsNum<ui32>(path + "/retry_settings/retries_on_start", 3);
        res.RetriesInBackground = c.AsNum<ui32>(path + "/retry_settings/retries_in_background", 2);
        res.WorkerAwakingPeriodSec = c.AsNum<ui32>(path + "/retry_settings/worker_awaking_period_sec", 10);
        res.DstsLimit = c.AsNum<ui32>(path + "/retry_settings/dsts_limit", 300);
        res.RolesUpdatePeriodSec = c.AsNum<ui32>(path + "/retry_settings/roles_update_period_sec", 600);
        res.RolesWarnPeriodSec = c.AsNum<ui32>(path + "/retry_settings/roles_warn_period_sec", 1200);

        return res;
    }

    TConfig::TTvmCertClientSettings TConfig::ReadTvmCertClient(const NXml::TConfig& c, const TString& path) {
        return TConfig::TTvmCertClientSettings{
            .Host = c.As<TString>(path + "/tvmcert/host", "localhost"),
            .Port = c.As<ui16>(path + "/tvmcert/port", 9001),
            .Timeout = TDuration::MilliSeconds(c.As<ui64>(path + "/tvmcert/timeout__msecs", 500)),
            .Workers = c.As<ui64>(path + "/tvmcert/workers", 32),
        };
    }

    std::vector<ui32> TConfig::ReadClientIds(const TString& clients) {
        std::vector<ui32> vec;

        NUtils::Transform(
            clients,
            ",;: ",
            [&vec, clients](const TStringBuf buf) -> void {
                if (buf) {
                    ui32 res = 0;
                    Y_ENSURE(TryIntFromString<10>(buf, res),
                             "Bad number '" << buf << "' in '" << clients << "'");
                    vec.push_back(res);
                }
            });

        return vec;
    }
}
