#pragma once

#include "config.h"

#include <passport/infra/daemons/tvmapi/src/utils/client.h>

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/tvm/common/decryptor.h>
#include <passport/infra/libs/cpp/unistat/diff.h>
#include <passport/infra/libs/cpp/utils/shared_state.h>

#include <library/cpp/tvmauth/deprecated/service_context.h>
#include <library/cpp/tvmauth/src/rw/keys.h>

#include <util/generic/string.h>

#include <memory>
#include <set>
#include <unordered_map>

namespace NPassport::NDbPool {
    class TDbPool;
}

namespace NPassport::NUnistat {
    class TBuilder;
}

namespace NPassport::NTvm {
    struct TCryptoKey {
        TCryptoKey(ui32 id)
            : Id(id)
        {
        }

        bool operator==(const TCryptoKey& o) const {
            return Id == o.Id;
        }

        ui32 Id;
        TString PublicKey;
        TString PrivateKey;
        time_t Ts = 1;
        bool IsDeleted = false;

        void ThrowIfInvalid(NTvmAuth::TTvmId clientId) const;
    };
    using TCryptoKeys = std::unordered_map<ui32, TCryptoKey>;

    using TClientPtr = std::shared_ptr<TClient>;
    using TClients = std::unordered_map<ui32, TClientPtr>;
    using TClientsPtr = std::shared_ptr<TClients>;
    using TRwPrivateKeyPtr = std::shared_ptr<NTvmAuth::NRw::TRwPrivateKey>;
    using TPrivateKeysPtr = std::shared_ptr<std::unordered_map<TClientId, TString>>;
    using TServiceCtxPtr = std::shared_ptr<NTvmAuth::TServiceContext>;
    using TSecretIndex = std::unordered_multimap<TString, TClientPtr>;
    using TSecretIndexPtr = std::shared_ptr<TSecretIndex>;

    struct TDbCache {
        TClientsPtr Clients;
        std::shared_ptr<const TString> PublicTvmKeys;
        TRwPrivateKeyPtr TvmKey;
        TPrivateKeysPtr PrivateBbKeys;
        TServiceCtxPtr ServiceCtx;
        TSecretIndexPtr SecretIndex;
        time_t KeysAge = 0;
    };

    class TDbCacheBuilder {
    public:
        TDbCacheBuilder(const NTvmCommon::TDecryptor& decr, NUnistat::TSignalDiff<>& errors, size_t minKeyCount = 0);

        void AddClientAttr(TClientId id, int type, const TString& val);
        void AddKeyAttr(ui32 id, int type, const TString& val);

        void PostProcess();
        void Serialize(ui32 preferedIdx, const TConfig::TPassportIds& passpIds);
        TDbCache Finalize();

        static void FillClientName(TClient& cl, const TString& str);
        static void FillClientSecret(TClient& cl, const TString& str, const NTvmCommon::TDecryptor& decr);
        static void FillKeyIds(TClient& cl, const TString& str);
        static void FillClientOldSecret(TClient& cl, const TString& str, const NTvmCommon::TDecryptor& decr);
        static void FillAbcId(TClient& cl, const TString& str);

        static TString GetBinaryKey(const TString& value,
                                    ui32 id,
                                    const NTvmCommon::TDecryptor& decr);

        void SetTvmPrivateKey(TDbCache& cache,
                              const TCryptoKeys& keys,
                              TClientId tvmId,
                              ui32 preferedIdx);
        void SerializePassportKeys(TDbCache& cache,
                                   const TCryptoKeys& keys,
                                   const TConfig::TPassportIds& passpIds,
                                   size_t minimunKeyCount);
        static void BuildSecretIndex(TDbCache& cache);

        const TClients& Clients() const {
            return Clients_;
        }

        const TCryptoKeys& Keys() const {
            return Keys_;
        }

    private:
        TClients Clients_;
        TCryptoKeys Keys_;
        std::set<ui32> DeletedKeyId_;

        TDbCache Result_;

        const NTvmCommon::TDecryptor& Decr_;
        const size_t MinKeyCount_;
        NUnistat::TSignalDiff<>& Errors_;
    };

    class TDbFetcher {
    public:
        TDbFetcher(const TConfig& config,
                   NDbPool::TDbPool& db,
                   const TString& keyFile,
                   const TString& diskCache,
                   size_t minKeyCount);
        virtual ~TDbFetcher() = default;

        void AddUnistat(NUnistat::TBuilder& builder) const;

        struct TResult {
            TResult() = default;
            TResult(TClientPtr client, TRwPrivateKeyPtr tvmKey)
                : Client_(std::move(client))
                , TvmKey_(std::move(tvmKey))
            {
            }

            TResult(const TResult&) = default;
            TResult& operator=(const TResult&) = default;
            TResult(TResult&&) = default;
            TResult& operator=(TResult&&) = default;

            explicit operator bool() const {
                return bool(Client_);
            }

            const TClient& Client() const {
                return *Client_;
            }

            const NTvmAuth::NRw::TRwPrivateKey& Key() const {
                return *TvmKey_;
            }

            bool IsKeyOk() const {
                return bool(TvmKey_);
            }

        private:
            TClientPtr Client_;
            TRwPrivateKeyPtr TvmKey_;
        };
        TResult GetClient(TClientId id) const;

        std::shared_ptr<const TString> PublicTvmKeys() const;
        TRwPrivateKeyPtr PrivateKey() const;
        TPrivateKeysPtr PrivateBbKeys() const;
        TServiceCtxPtr ServiceCtx() const;
        TSecretIndexPtr SecretIndex() const;
        time_t KeysAge() const;

    protected:
        void Run();

    private:
        /*
         * Full reload of clients' info with keys periodically
         */
        using TCachePtr = std::shared_ptr<TDbCache>;

        TCachePtr Load() const;

        void LoadFromFile();

        NUtils::TSharedState<TDbCache> Cache_;
        const TString DiskCache_;

        mutable NUnistat::TSignalDiff<> UnistatQueryErrors_ = {"internal_error.dbfetcher.query"};
        mutable NUnistat::TSignalDiff<> UnistatParsingErrors_ = {"internal_error.dbfetcher.parsing"};

        NDbPool::TDbPool& Db_;
        NTvmCommon::TDecryptor Decryptor_;
        const TConfig::TDbFetcher Conf_;
        const TConfig::TPassportIds PasspIds_;
        const size_t MinKeyCount_;
    };
}
