#include "notify_log.h"

#include <passport/infra/libs/cpp/utils/thread_local_id.h>
#include <passport/infra/libs/cpp/utils/log/file_logger.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/string_utils/tskv_format/builder.h>

namespace NPassport::NTvm {
    class TNumCaster {
    public:
        template <typename T>
        TStringBuf operator()(T num) {
            static_assert(std::is_integral<T>::value, "allow only integral types");
            return TStringBuf(CastBuffer_, IntToString<10>(num, CastBuffer_, sizeof(CastBuffer_)));
        }

    private:
        char CastBuffer_[48];
    };

    TNotifyLog::TBaseEntry::TBaseEntry(const TDbFetcher::TResult& src, const TString& ip)
        : Src(src.Client().Id())
        , SrcIp(ip)
        , SrcName(src.Client().Name())
        , SrcAbcId(src.Client().GetAbcId())
    {
    }

    TNotifyLog::TDeletedDst::TDeletedDst(const TDbFetcher::TResult& src,
                                         const TString& ip,
                                         const TDbFetcher::TResult& dst)
        : TBaseEntry(src, ip)
        , Dst(dst.Client().Id())
        , DstName(dst.Client().Name())
        , DstAbcId(dst.Client().GetAbcId())
    {
    }

    TNotifyLog::TNotifyLog(const TString& filename)
        : Logger_(std::make_unique<NUtils::TFileLogger>(filename, "ERROR", false))
    {
    }

    TNotifyLog::~TNotifyLog() = default;

    void TNotifyLog::Log(const TNotifyLog::TDeletedSrc& entry, TInstant now) const {
        Logger_->Log(BuildRow(entry, now));
    }

    void TNotifyLog::Log(const TNotifyLog::TDeletedDst& entry, TInstant now) const {
        Logger_->Log(BuildRow(entry, now));
    }

    void TNotifyLog::Log(const TNotifyLog::TOldSecret& entry, TInstant now) const {
        Logger_->Log(BuildRow(entry, now));
    }

    TString TNotifyLog::BuildRow(const TNotifyLog::TDeletedSrc& entry, TInstant now) {
        NTskvFormat::TLogBuilder builder = CreateBuilder(now);

        builder.AddUnescaped("type", "deleted_src");
        AddBase(builder, entry);

        return builder.End().Str();
    }

    TString TNotifyLog::BuildRow(const TNotifyLog::TDeletedDst& entry, TInstant now) {
        NTskvFormat::TLogBuilder builder = CreateBuilder(now);

        builder.AddUnescaped("type", "deleted_dst");
        AddBase(builder, entry);
        builder
            .AddUnescaped("dst", TNumCaster()(entry.Dst))
            .AddUnescaped("dst_name", entry.DstName)
            .AddUnescaped("dst_abc", entry.DstAbcId
                                         ? TNumCaster()(*entry.DstAbcId)
                                         : "");

        return builder.End().Str();
    }

    TString TNotifyLog::BuildRow(const TNotifyLog::TOldSecret& entry, TInstant now) {
        NTskvFormat::TLogBuilder builder = CreateBuilder(now);

        builder.AddUnescaped("type", "old_secret");
        AddBase(builder, entry);

        return builder.End().Str();
    }

    void TNotifyLog::AddBase(NTskvFormat::TLogBuilder& builder, const TNotifyLog::TBaseEntry& entry) {
        builder
            .AddUnescaped("src", TNumCaster()(entry.Src))
            .AddUnescaped("src_name", entry.SrcName)
            .AddUnescaped("src_ip", entry.SrcIp)
            .AddUnescaped("src_abc", entry.SrcAbcId
                                         ? TNumCaster()(*entry.SrcAbcId)
                                         : "");
    }

    NTskvFormat::TLogBuilder TNotifyLog::CreateBuilder(TInstant now) {
        NTskvFormat::TLogBuilder builder;
        builder.Str().reserve(1024);
        builder.Begin("tvm-notify", now.Seconds());
        builder.AddUnescaped("datetime", now.ToStringLocal());
        builder.AddUnescaped("request_id", NUtils::GetThreadLocalRequestId());
        return builder;
    }
}
