#include "oauth_client.h"

#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/stream/str.h>

namespace NPassport::NTvm {
    TOAuthClient::TOAuthClient(const TConfig::TOAuth& cfg)
        : Handle_(CreateClient(cfg))
    {
    }

    TString TOAuthClient::GetToken(const TConfig::TOAuth& cfg) {
        TString post = NUtils::CreateStr(
            "grant_type=password&username=", NUtils::Urlencode(cfg.Login),
            "&password=", NUtils::Urlencode(cfg.Pwd),
            "&client_id=", NUtils::Urlencode(cfg.ClientId),
            "&client_secret=", NUtils::Urlencode(cfg.ClientSecret));

        TStringStream s;
        const auto code = Handle_.DoPost("/token", post, &s);
        Y_ENSURE(200 == code, "Failed to get oauth token (" << code << "):" << s.Str());

        TString token = ParseResponse(s.Str());
        TLog::Info() << "OAuth token successfully fetched from " << cfg.Host;
        return token;
    }

    TString TOAuthClient::ParseResponse(const TStringBuf body) {
        rapidjson::Document doc;
        Y_ENSURE(NJson::TReader::DocumentAsObject(body, doc),
                 "Sshkey: can't parse json from Oauth: main object. " << body);

        TString t;
        Y_ENSURE(NJson::TReader::MemberAsString(doc, "access_token", t) && !t.empty(),
                 "Sshkey: can't parse json from Oauth: access_token. " << body);

        return t;
    }

    TKeepAliveHttpClient TOAuthClient::CreateClient(const TConfig::TOAuth& cfg) {
        TStringStream url;
        url << cfg.Schema << "://" << cfg.Host;
        ui16 port = cfg.Schema == "https" ? 443 : 80;
        return TKeepAliveHttpClient(url.Str(), port);
    }
}
