#include "runtime_context.h"

#include "abc_fetcher.h"
#include "dbfetcher.h"
#include "notify_log.h"
#include "oauth_client.h"
#include "staff_fetcher.h"
#include "tvmcert_client.h"

#include <passport/infra/daemons/tvmapi/src/pregeneration/facade.h>
#include <passport/infra/daemons/tvmapi/src/utils/retry_settings.h>
#include <passport/infra/daemons/tvmapi/src/utils/utils.h>

#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/regular_task.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <util/system/fs.h>

#include <bitset>

namespace NPassport::NTvm {
    void TRuntimeContext::TDb::AddUnistat(NUnistat::TBuilder& builder) const {
        Ctx_->AddUnistat(builder);
    }

    TRuntimeContext::TRuntimeContext(const NXml::TConfig& config, const TString& path)
        : Config_(TConfig(config, path))
    {
        InitDb(config, path);

        if (config.AsBool(path + "/secure_heap/enabled", false)) {
            const size_t secureHeapSize = config.AsNum<size_t>(
                path + "/secure_heap/size",
                256 * 1024 * 1024);
            Y_ENSURE(std::bitset<64>(secureHeapSize).count() == 1,
                     "secure_heap size must be > 0 and be power of 2");
            const int secureHeapMinSize = config.AsInt(
                path + "/secure_heap/minsize",
                16);
            Y_ENSURE(std::bitset<32>(secureHeapMinSize).count() <= 1,
                     "secure_heap minsize must be power of 2");
            NTvmAuth::NRw::TSecureHeap::Init(secureHeapSize, secureHeapMinSize);
        }

        DbFetcher_ = NUtils::TRegularTaskDecorator<TDbFetcher>::CreateUnique(
            {
                TDuration::Seconds(Config_.DbFetcher.Period),
                "refresh_db",
            },
            Config_,
            *Db_.Tvm(),
            Config_.DbFetcher.KeyFile,
            Config_.DbFetcher.DiskCache,
            Config_.DbFetcher.MinKeyCount);

        if (Config_.Staff.Enabled) {
            StaffFetcher_ = NUtils::TRegularTaskDecorator<TStaffFetcher>::CreateUnique(
                {
                    TDuration::Seconds(Config_.Staff.RefreshPeriod),
                    "staff_refresher",
                },
                *this);
            if (Config_.Abc.Enabled) {
                AbcFetcher_ = NUtils::TRegularTaskDecorator<TAbcFetcher>::CreateUnique(
                    {
                        TDuration::Seconds(Config_.Abc.RefreshPeriod),
                        "abc_refresher",
                    },
                    *this);
            }
        }

        Pregeneration_ = NUtils::TRegularTaskDecorator<NPregen::TFacade>::CreateUnique(
            {
                TDuration::Seconds(Config_.Pregen.Period),
                "pregen",
            },
            *this);

        NotifyLog_ = std::make_unique<TNotifyLog>(config.AsString(path + "/log_notify"));

        RetrySettings_ = std::make_unique<TRetrySettingsHelper>(Config_.RetrySettings);

        TvmCertClient_ = std::make_unique<TTvmCertClient>(*this);
    }

    TRuntimeContext::~TRuntimeContext() = default;

    void TRuntimeContext::AddUnistat(NUnistat::TBuilder& builder) const {
        Db_.AddUnistat(builder);
        Pregeneration_->AddUnistat(builder);
        DbFetcher_->AddUnistat(builder);
        if (StaffFetcher_) {
            StaffFetcher_->AddUnistat(builder);
        }
        if (AbcFetcher_) {
            AbcFetcher_->AddUnistat(builder);
        }
        builder.Add(UnistatOauthErrors_);
    }

    bool TRuntimeContext::IsForceDown() const {
        return !Config_.Gen.ForceDownFile.empty() &&
               NFs::Exists(Config_.Gen.ForceDownFile) &&
               DbFetcher_->PrivateKey();
    }

    static const TString MSG_TVM = "No db pool for tvm db";
    bool TRuntimeContext::IsDbOk(TString& str) const {
        // Tvm db
        if (!Db_.Tvm()) {
            str = MSG_TVM;
            return false;
        }

        return Db_.Tvm()->IsOk(&str);
    }

    bool TRuntimeContext::IsVerifierSshClientIdAllowed(ui32 clId) const {
        return std::find(Config_.Gen.VerifySshClientIds.begin(),
                         Config_.Gen.VerifySshClientIds.end(),
                         clId) != Config_.Gen.VerifySshClientIds.end();
    }

    bool TRuntimeContext::IsCheckSecretClientIdAllowed(ui32 clId) const {
        return std::find(Config_.Gen.CheckSecretClientIds.begin(),
                         Config_.Gen.CheckSecretClientIds.end(),
                         clId) != Config_.Gen.CheckSecretClientIds.end();
    }

    TString TRuntimeContext::GetOAuthToken() const {
        TErrorIncrementer err{&UnistatOauthErrors_};
        TString token = Auth_.Get(Config_.Oauth);
        err.Err = nullptr;
        return token;
    }

    const TRetrySettingsHelper& TRuntimeContext::GetRetrySettingsHelper() const {
        return *RetrySettings_;
    }

    void TRuntimeContext::InitDb(const NXml::TConfig& config, const TString& path) {
        Db_.Ctx_ = NDbPool::TDbPoolCtx::Create();
        Db_.Ctx_->InitLogger(config.AsString(path + "/log_dbpool"));

        Db_.Tvm_ = Db_.Ctx_->CreateDb(config, path + "/tvm_db");
        try {
            Db_.Tvm_->TryPing();
        } catch (const std::exception& e) {
            TLog::Error() << "Can't create dbpool for tvm db. Disk cache is our straw. " << e.what();
        }
    }

    TString TRuntimeContext::TAuthorizationHolder::Get(const TConfig::TOAuth& cfg) const {
        {
            std::shared_lock lock(Mutex_);
            if (!Header_.empty()) {
                return Header_;
            }
        }

        std::unique_lock lock(Mutex_);
        if (Header_.empty()) {
            Header_ = TOAuthClient(cfg).GetToken(cfg); // can throw
        }
        return Header_;
    }

}
