#include "tvmcert_client.h"

#include "runtime_context.h"

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/handle.h>
#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/http/misc/httpcodes.h>

namespace NPassport::NTvm {
    TTvmCertClient::TTvmCertClient(const TRuntimeContext& runtime)
        : Runtime_(runtime)
    {
        Pool_ = Runtime_.Db().DbPoolCtx()->CreateDb(NDbPool::TDbPoolSettings{
            .Dsn = NDbPool::TDestination::CreateHttp("tvmcert"),
            .Hosts = std::vector<NDbPool::TDbHost>{
                NDbPool::TDbHost{
                    .Host = Runtime_.Config().TvmCertClientSettings.Host,
                    .Port = Runtime_.Config().TvmCertClientSettings.Port,
                },
            },
            .Size = Runtime_.Config().TvmCertClientSettings.Workers,
            .QueryTimeout = Runtime_.Config().TvmCertClientSettings.Timeout,
        });

        try {
            Pool_->TryPing();
        } catch (const NDbPool::TException& e) {
            TLog::Debug() << "Cannot connect to tvmcert: %s" << e.what();
        }
    }

    TTvmCertClient::TCheckResult TTvmCertClient::CheckSign(const TStringBuf requestID, const TStringBuf login, const TStringBuf cert, const TStringBuf sign, const TStringBuf rawString) const {
        NDbPool::TQuery query("/1/verify");
        query.SetHttpMethod("POST");
        query.SetHttpBody(NUtils::CreateStr(
            "username=", NUtils::Urlencode(login),
            "&certificate=", NUtils::Urlencode(cert),
            "&sign=", NUtils::Urlencode(sign),
            "&data=", NUtils::Urlencode(rawString),
            "&request_id=", requestID));

        try {
            NDbPool::TBlockingHandle h(*Pool_);
            std::unique_ptr<NDbPool::TResult> result = h.Query(std::move(query));
            NDbPool::THttpResponse response = result->ToHttpResponse();
            if (response.Status != HTTP_OK) {
                rapidjson::Document doc;
                TString status;
                TString message;
                if (!NJson::TReader::DocumentAsObject(response.Body, doc) ||
                    !NJson::TReader::MemberAsString(doc, "status", status) ||
                    !NJson::TReader::MemberAsString(doc, "message", message)) {
                    TLog::Error() << "Cannot parse response body from tvmcert: [" << response.Status << "] " << response.Body;
                    return TCheckResult{
                        .Err = "Unparsable error",
                    };
                }

                TLog::Debug() << "Error checking certificate: [" << response.Status << " - " << status << "] " << message;
                return TCheckResult{
                    .Err = status,
                };
            }

            return {};
        } catch (const std::exception& e) {
            TLog::Debug() << "Failed to fetch keys from tvmcert: " << e.what();
            return TCheckResult{
                .Err = "Failed send request to tvmcert",
            };
        }
    }
}
