#include "top_consumers.h"

#include <passport/infra/libs/cpp/unistat/absolute.h>
#include <passport/infra/libs/cpp/unistat/builder.h>

namespace NPassport::NTvm::NTopConsumers {
    TConsumer::TConsumer(const TString& name, TDuration slidingWindow)
        : UnistatSignalName_("top_consumers." + name, NUnistat::NSuffix::AXXX)
    {
        size_t count = slidingWindow.Seconds();
        Y_VERIFY(count > 0 && count < 100, "got %s", slidingWindow.ToString().c_str());

        for (size_t idx = 0; idx < count; ++idx) {
            Data_.push_back(std::make_unique<TNode>());
        }
    }

    void TConsumer::Inc(TInstant now) {
        const time_t t = now.TimeT();

        TNode& n = *Data_[t % Data_.size()];

        // Yes, it is racy way to collect rps
        // We don't need to collect rps here: we need to collect weight of consumer
        // It is ok to loose some hits
        const time_t prevT = n.LastAccess.exchange(t);
        if (t == prevT) {
            ++n.Value;
        } else {
            n.Value.StoreValue(1);
        }
    }

    ui64 TConsumer::Get(TInstant now) {
        const time_t useFrom = now.TimeT() - Data_.size();
        ui64 sum = 0;

        for (const TNodePtr& n : Data_) {
            if (useFrom < n->LastAccess.GetValue()) {
                sum += n->Value.GetValue();
            }
        }

        return sum;
    }

    const TString& TConsumer::UnistatSignalName() const {
        return UnistatSignalName_.GetName();
    }

    TTopConsumers::TTopConsumers(const TTopConsumersSettings& settings)
        : Settings_(settings)
    {
        Storage_.reserve(10000);
    }

    void TTopConsumers::Add(const TString& name, TInstant now) {
        {
            std::shared_lock lock(Mutex_);
            auto it = Storage_.find(name);
            if (it != Storage_.end()) {
                it->second->Inc(now);
                return;
            }
        }

        std::unique_lock lock(Mutex_);
        auto [it, inserted] = Storage_.emplace(name, nullptr);
        if (inserted) {
            it->second = std::make_shared<TConsumer>(name, Settings_.SlidingWindow);
        }

        it->second->Inc(now);
    }

    void TTopConsumers::AddUnistat(NUnistat::TBuilder& builder, TInstant now) const {
        struct TTmp {
            TConsumerPtr Consumer;
            ui64 Value = 0;

            bool operator<(const TTmp& o) const {
                return Value > o.Value;
            }
        };
        std::vector<TTmp> consumers;

        {
            std::shared_lock lock(Mutex_);
            for (const auto& [name, c] : Storage_) {
                consumers.push_back({.Consumer = c});
            }
        }

        for (TTmp& t : consumers) {
            t.Value = t.Consumer->Get(now);
        }

        const size_t toShow = std::min(Settings_.CountToShow, consumers.size());
        std::partial_sort(consumers.begin(),
                          consumers.begin() + toShow,
                          consumers.end());

        for (size_t idx = 0; idx < toShow; ++idx) {
            const TTmp& t = consumers[idx];
            builder.AddRow(t.Consumer->UnistatSignalName(), 1);
        }
    }
}
