#include "tvm.h"

#include "exception.h"
#include "output/serializer.h"
#include "output/status.h"
#include "runtime_context/experiment.h"
#include "runtime_context/strings.h"
#include "stats/top_consumers.h"

#include <passport/infra/libs/cpp/dbpool/db_pool_stats.h>
#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/log/file_logger.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <util/generic/strbuf.h>
#include <util/string/cast.h>

namespace NPassport::NTvm {
    TTvm::TTvm() {
        auto addPath = [this](const TString& path, auto handle) {
            V2Handlers_.emplace(path, [this, handle](NCommon::TRequest& req) { (this->*handle)(req); });
            Unistat_.Path.Add(path, "path." + path);
        };
        addPath("/2/check_secret", &TTvm::HandleV2CheckSecret);
        addPath("/2/keys", &TTvm::HandleV2Keys);
        addPath("/2/private_keys", &TTvm::HandleV2PrivateKeys);
        addPath("/2/ticket", &TTvm::HandleV2Ticket);
        addPath("/2/verify_ssh", &TTvm::HandleV2VerifySsh);

        auto addGrantType = [this](const TString& path, auto handle) {
            GrantTypeHandlers_.emplace(path, [this, handle](NCommon::TRequest& req) { return (this->*handle)(req); });
            Unistat_.GrantType.Add(path, "grant_type." + path);
        };
        addGrantType("client_credentials", &TTvm::HandleGtClientCredentials);
        addGrantType("sshkey", &TTvm::HandleGtSshKey);
    }

    TTvm::~TTvm() = default;

    void TTvm::Init(const NXml::TConfig& config) {
        try {
            const TString configPath = "/config/components/component";

            // common
            InitLogger(config, configPath);
            InitProcessors(config, configPath);
            InitExperiments(config, configPath);
            InitMisc(config, configPath);

            TLog::Info() << "onLoad successfully finished";
        } catch (const std::exception& e) {
            TLog::Error() << "Tvm init() failed: " << e.what();
            throw;
        }
    }

    static const TString ERR_ARG_REQUIRED = "Arg is required but empty: ";
    static const TString FATAL_EXCEPTION = "Fatal exception: ";
    static const TString SOMETHING_WRONG_MESSAGE = "Something went wrong";
    void TTvm::HandleRequest(NCommon::TRequest& req) {
        req.ScanCgiFromBody();

        auto start = TInstant::Now();

        try {
            ChooseMethod(req);
        } catch (const TIncorrectArgException& e) {
            Proc4Xx(req);
            TSerializer::Serialize(req, TResult(e.what(), TStatus(TStatus::STR_ERR_REQUEST, TStatus::STR_INCORRECT_ + e.WhatArg())));
            TLog::Debug() << "Request failed. Incorrect arg: " << e.WhatArg();
        } catch (const TMissingArgException& e) {
            Proc4Xx(req);
            TSerializer::Serialize(req, TResult(ERR_ARG_REQUIRED + e.what(), TStatus(TStatus::STR_ERR_REQUEST, TStatus::STR_MISSING_ + e.what())));
            TLog::Debug() << "Request failed. Missing arg: " << e.what();
        } catch (const TException& e) {
            TSerializer::Serialize(req, TResult(FATAL_EXCEPTION + SOMETHING_WRONG_MESSAGE, TStatus::ERROR_FATAL));
            TLog::Error() << FATAL_EXCEPTION << e.what();
        } catch (const std::exception& e) {
            TSerializer::Serialize(req, TResult(FATAL_EXCEPTION + SOMETHING_WRONG_MESSAGE, TStatus::ERROR_FATAL));
            TLog::Error() << FATAL_EXCEPTION << e.what();
        } catch (...) {
            TString msg = FATAL_EXCEPTION + "Totally unexpected";
            TSerializer::Serialize(req, TResult(msg, TStatus::ERROR_FATAL));
            TLog::Error() << "Fatal error (WTF): " << msg;
        }

        auto end = TInstant::Now();
        LogAccess(req, end - start);
    }

    void TTvm::AddUnistat(NUnistat::TBuilder& builder) {
        Runtime_->AddUnistat(builder);
        Unistat_.Path.AddUnistat(builder);
        Unistat_.GrantType.AddUnistat(builder);
        Unistat_.TopConsumers->AddUnistat(builder);
    }

    void TTvm::AddUnistatExtended(const TString& path, NUnistat::TBuilder& builder) const {
        TStringBuf p = path;
        p.ChopSuffix("/");

        if (p == "/consumers") {
            Unistat_.Consumers.AddUnistat(builder);
        } else if (p == "/consumers_4XX") {
            Unistat_.ConsumersWith4Xx.AddUnistat(builder);
        } else if (p == "/consumers_old_secret") {
            Unistat_.ConsumersWithOld.AddUnistat(builder);
        }
    }

    static const TString SCRIPT_IS_UNKNOWN = "Script is unknown: '";
    void TTvm::ChooseMethod(NCommon::TRequest& req) {
        CheckDuplicatedArgs(req);

        TStringBuf path = req.GetPath();
        path.ChopSuffix("/");
        Unistat_.Path.Inc(path);

        if (path.Contains("/nagios") || path == "/ping") {
            ProcessPing(req);
            return;
        }

        if (path.StartsWith("/2") && ProcessApiV2(path, req)) {
            return;
        }

        TString msg = NUtils::CreateStr(SCRIPT_IS_UNKNOWN, req.GetPath(), "\'");
        TSerializer::Serialize(req, TResult(msg, TStatus::ERROR_REQUEST__MISSING));
        TLog::Debug() << "Request failed. " << msg;
    }

    static const TString GRANT_TYPE = "grant_type";
    static const TString SRC = "src";
    static const TString LOGIN = "login";
    static const TString UID = "uid";
    static const TString UNKNOWN = "_unknown_";
    bool TTvm::ProcessApiV2(const TStringBuf path, NCommon::TRequest& req) {
        auto it = V2Handlers_.find(path);
        if (it != V2Handlers_.end()) {
            it->second(req);
            return true;
        }

        return false;
    }

    static const TString POST_ = "POST";
    bool TTvm::CheckOnlyPost(NCommon::TRequest& req) {
        if (POST_ != req.GetRequestMethod() || !req.GetQueryString().empty()) {
            static const TResult RES_POST_ONLY("All query args must be sent only in POST body", TStatus::ERROR_REQUEST__POST_ONLY);
            TSerializer::Serialize(req, RES_POST_ONLY);
            TLog::Debug() << "Request failed. POST only";
            return false;
        }

        return true;
    }

    static const TString GET_ = "GET";
    bool TTvm::CheckOnlyGet(NCommon::TRequest& req) {
        if (GET_ != req.GetRequestMethod() || !req.IsBodyEmpty()) {
            static const TResult RES_GET_ONLY("All query args must be sent only in GET query", TStatus::ERROR_REQUEST__GET_ONLY);
            TSerializer::Serialize(req, RES_GET_ONLY);
            TLog::Debug() << "Request failed. GET only";
            return false;
        }

        return true;
    }

    void TTvm::CheckDuplicatedArgs(const NCommon::TRequest& req) {
        NCommon::TRequest::TDuplicatedArgs dups = req.GetDuplicatedArgs();
        if (dups.empty()) {
            return;
        }

        // TODO: always throw error: PASSP-32036

        TString buf;
        for (auto [key, value] : dups) {
            if (key == "lib_version" && req.GetPath().StartsWith("/2/keys")) {
                NUtils::AppendSeparated(buf, ',', NUtils::CreateStr(key, "->", value));
            } else {
                throw TIncorrectArgException(TString(key))
                    << "Duplicate args not allowed, got several values of '" << key << "'";
            }
        }

        TLog::Warning() << "Duplicated args: " << buf;
    }

    void TTvm::InitLogger(const NXml::TConfig& config, const TString& path) {
        config.InitCommonLog(path + "/logger_common");
        TLog::Info() << "Logger inited. Tvm is starting";

        if (config.Contains(path + "/logger_access")) {
            AccessLogger_ = config.CreateLogger(path + "/logger_access");
        } else {
            TLog::Info() << "Access log is not configured";
        }
    }

    void TTvm::InitProcessors(const NXml::TConfig& config, const TString& path) {
        Runtime_ = std::make_unique<TRuntimeContext>(config, path);

        V2_.KeysProcessor = std::make_unique<NV2::TKeysProcessor>(*Runtime_);
        V2_.ClCredProcessor = std::make_unique<NV2::TClCredProcessor>(
            *Runtime_,
            Unistat_.ConsumersWithOld);
        V2_.PrivKeysProcessor = std::make_unique<NV2::TPrivKeysProcessor>(*Runtime_);
        if (Runtime_->Config().Staff.Enabled && Runtime_->Config().Abc.Enabled) {
            V2_.SshkeyProcessor = std::make_unique<NV2::TSshkeyProcessor>(*Runtime_);
        }
        V2_.CheckSecretProcessor = std::make_unique<NV2::TCheckSecretProcessor>(*Runtime_);
    }

    void TTvm::InitExperiments(const NXml::TConfig& config, const TString& path) {
        Y_UNUSED(config);
        Y_UNUSED(path);
    }

    void TTvm::InitMisc(const NXml::TConfig& config, const TString& path) {
        size_t countToShow = config.AsInt(path + "/unistat/top_consumers/count_to_show", 100);
        size_t slidingWindow = config.AsInt(path + "/unistat/top_consumers/sliding_window__sec", 10);

        Y_ENSURE(slidingWindow > 0 && slidingWindow < 100, "got " << slidingWindow);

        using namespace NTopConsumers;

        Unistat_.TopConsumers = std::make_unique<TTopConsumers>(
            TTopConsumersSettings{
                .CountToShow = countToShow,
                .SlidingWindow = TDuration::Seconds(slidingWindow),
            });
    }

    static const TString HEADER_CONTENT_TYPE = "Content-Type";
    static const TString MIME_HTML = "text/html";
    static const TString DBPOOLSTATS("dbpoolstats");
    static const TString KEYS_AGE = "keys_age";
    static const TString TVM_FORCED_DOWN =
        "\r\n<html><head><title>Tvm monitoring status</title></head><body><h2>Not operational: externally forced out-of-service</h2></body></html>\r\n";
    static const TString PING_ERR_HEAD_STR =
        "\r\n<html><head><title>Tvm monitoring status</title></head><body><h2>Error: databases: ";
    static const TString PING_ERR_TAIL_STR =
        "</h2></body></html>\r\n";
    static const TString PING_OK =
        "\r\n<html><head><title>Tvm monitoring status</title></head><body><h2>OK</h2></body></html>\r\n";
    void TTvm::ProcessPing(NCommon::TRequest& req) const {
        // Keys age
        const time_t keysAge = Runtime_->DbFetcher().KeysAge();
        if (req.HasArg(KEYS_AGE)) {
            req.SetStatus(HTTP_OK);
            req.Write(TStringBuilder() << keysAge << Endl);
            return;
        }

        req.SetHeader(HEADER_CONTENT_TYPE, MIME_HTML);

        // Force down state
        if (Runtime_->IsForceDown()) {
            TLog::Info() << "tvm ping error: externally forced out-of-service";
            req.SetStatus(HTTP_SERVICE_UNAVAILABLE);
            req.Write(TVM_FORCED_DOWN);
            return;
        }

        // Db pool monitoring
        if (req.HasArg(DBPOOLSTATS)) {
            NDbPool::TDbPoolStats st;
            st.Add(Runtime_->Db().Tvm());
            req.SetStatus(HTTP_OK);
            req.Write(TString(st.Result()));
            return;
        }

        // Db pool status
        TString error;
        if (!Runtime_->IsDbOk(error)) {
            req.SetStatus(HTTP_INTERNAL_SERVER_ERROR);
            req.Write(NUtils::CreateStr(
                PING_ERR_HEAD_STR,
                error,
                PING_ERR_TAIL_STR));
            return;
        }

        // Keys age - one more time
        const time_t now = time(nullptr);
        const time_t ageDiff = now - keysAge;
        if (ageDiff > Runtime_->Config().Gen.KeysAcceptableAge) {
            TStringStream s;
            s << "Keys are too old. Age=" << keysAge << "."
              << " Now=" << now << "."
              << " Diff=" << ageDiff << "."
              << " AcceptableAge=" << Runtime_->Config().Gen.KeysAcceptableAge
              << Endl;
            req.SetStatus(HTTP_INTERNAL_SERVER_ERROR);
            req.Write(s.Str());
            return;
        }

        // OK. Other monitor features
        req.SetStatus(HTTP_OK);
        req.Write(PING_OK);
    }

    void TTvm::LogAccess(const NCommon::TRequest& req, TDuration duration) const {
        if (!AccessLogger_) {
            return;
        }

        TString cgiParams;
        std::vector<TString> args;
        req.ArgNames(args);
        if (args.empty()) {
            cgiParams.assign("-");
        } else {
            cgiParams.reserve(25 * args.size());
            for (const TString& arg : args) {
                if (!cgiParams.empty()) {
                    cgiParams.push_back('&');
                }

                cgiParams.append(arg).push_back('=');
                const TString& val = req.GetArg(arg);

                if (val.length() > 255 && arg != TStrings::DST_) {
                    cgiParams.append(val, 0, 255).append("...");
                } else if ((arg == "oauth_token" || arg.EndsWith("sign")) && !val.empty()) {
                    cgiParams.append(val, 0, val.size() / 2 + 1).append("...");
                } else {
                    cgiParams.append(val);
                }
            }
        }
        NUtils::EscapeUnprintable(cgiParams);

        AccessLogger_->Info("%s http%s %s %.1f %d %s %s",
                            req.GetRequestId().c_str(),
                            req.IsSecure() ? "s" : "",
                            req.GetRemoteAddr().c_str(),
                            duration.MicroSeconds() / 1000.0,
                            req.GetStatusCode(),
                            req.GetPath().c_str(),
                            cgiParams.c_str());
    }

    void TTvm::Proc4Xx(const NCommon::TRequest& req) {
        ProcSignals(req, Unistat_.ConsumersWith4Xx);
    }

    template <typename T>
    void TTvm::ProcSignals(const NCommon::TRequest& req, T& consumers) {
        const TString& src = req.GetArg(SRC);
        if (src) {
            consumers.Add(src);
            return;
        }

        const TString& login = req.GetArg(LOGIN);
        const TString& uid = req.GetArg(UID);
        if (login || uid) {
            TStringBuf prefix;
            if (req.GetPath().Contains("verify_ssh")) {
                prefix = "verify_ssh";
            } else if (req.GetArg(GRANT_TYPE) == "sshkey") {
                prefix = "sshkey";
            }

            if (prefix) {
                consumers.Add(NUtils::CreateStr(prefix, ".", login ? login : uid));
                return;
            }
        }

        consumers.Add(UNKNOWN);
    }

    TResult TTvm::GrantTypeIsUnknown(const TString& grantType) {
        TString msg = NUtils::CreateStr("Grant type is unknown: '", grantType, "'");
        TLog::Debug() << "Request failed. " << msg;
        return TResult(msg, TStatus::ERROR_REQUEST__MISSING);
    }

    void TTvm::HandleV2CheckSecret(NCommon::TRequest& req) const {
        if (!CheckOnlyGet(req)) {
            return;
        }

        TSerializer::Serialize(req, V2_.CheckSecretProcessor->Process(req));
    }

    void TTvm::HandleV2Keys(NCommon::TRequest& req) const {
        if (!CheckOnlyGet(req)) {
            return;
        }

        const TResult res = V2_.KeysProcessor->Process(req);

        TSerializer::Serialize(req, res);
        for (const auto& [name, value] : res.Headers()) {
            req.SetHeader(name, value);
        }
    }

    void TTvm::HandleV2PrivateKeys(NCommon::TRequest& req) const {
        if (!CheckOnlyPost(req)) {
            return;
        }

        TSerializer::Serialize(req, V2_.PrivKeysProcessor->Process(req));
    }

    void TTvm::HandleV2Ticket(NCommon::TRequest& req) {
        if (!CheckOnlyPost(req)) {
            return;
        }

        ProcSignals(req, Unistat_.Consumers);
        ProcSignals(req, *Unistat_.TopConsumers);

        const TString& grantType = req.GetArg(GRANT_TYPE);

        Unistat_.GrantType.Inc(grantType);

        auto it = GrantTypeHandlers_.find(grantType);
        const TResult res = it == GrantTypeHandlers_.end()
                                ? GrantTypeIsUnknown(grantType)
                                : it->second(req);
        if (res.Status().Is4Xx()) {
            Proc4Xx(req);
        }

        TSerializer::Serialize(req, res);
        for (const auto& [name, value] : res.Headers()) {
            req.SetHeader(name, value);
        }
    }

    void TTvm::HandleV2VerifySsh(NCommon::TRequest& req) {
        if (!CheckOnlyPost(req)) {
            return;
        }

        if (!V2_.SshkeyProcessor) {
            static const TResult STAFF_IS_DEAD("Staff is not configured", TStatus::ERROR_FATAL);
            TSerializer::Serialize(req, STAFF_IS_DEAD);
            TLog::Error() << "Request failed. Staff is not configured";
            return;
        }

        const TResult res = V2_.SshkeyProcessor->VerifySsh(req);
        if (res.Status().Is4Xx()) {
            Proc4Xx(req);
        }

        TSerializer::Serialize(req, res);
    }

    TResult TTvm::HandleGtClientCredentials(NCommon::TRequest& req) const {
        return V2_.ClCredProcessor->Process(req);
    }

    TResult TTvm::HandleGtSshKey(NCommon::TRequest& req) const {
        if (!V2_.SshkeyProcessor) {
            TLog::Error() << "Request failed. Abc is not configured";
            return TResult("Abc is not configured", TStatus::ERROR_FATAL);
        }

        return V2_.SshkeyProcessor->Process(req);
    }
}
