#include "client.h"

#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/json/writer.h>
#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/generic/yexception.h>

namespace NPassport::NTvm {
    TClient::TClient(TClientId id)
        : Id_(id)
    {
    }

    void TClient::AddKeyId(TKeyId keyId) {
        KeysIds_.push_back(keyId);
    }

    const std::vector<TKeyId>& TClient::KeysIds() const {
        return KeysIds_;
    }

    const TString& TClient::Secret() const {
        return Secret_;
    }

    const TString& TClient::OldSecret() const {
        return OldSecret_;
    }

    TClient::ECheckStatus TClient::CheckSecret(const TStringBuf sign, const TStringBuf rawString) const {
        if (Secret_.empty()) { // impossible now
            return ECheckStatus::Invalid;
        }

        if (NUtils::SecureCompare(sign, NUtils::TCrypto::HmacSha256(Secret_, rawString))) {
            return ECheckStatus::Valid;
        }

        bool res = !OldSecret_.empty() && NUtils::SecureCompare(sign, NUtils::TCrypto::HmacSha256(OldSecret_, rawString));
        if (res) {
            TLog::Debug() << "Tvm client uses old secret. id=" << Id_;
        }

        return res ? ECheckStatus::ValidWithOldSecret : ECheckStatus::Invalid;
    }

    bool TClient::operator==(const TClient& other) const {
        return Id_ == other.Id_;
    }

    void TClient::SetSecret(const TString& secret) {
        // Protection: we try to prevent reading of key as secret
        // Spy can copy encrypted value from key_table with (key_id == client_id) to client_attr_table
        // Keys are bigger then secret
        if (secret.size() != 16) {
            throw yexception() << "Secret size is not correct! Id: " << Id_;
        }
        Secret_ = secret;
    }

    void TClient::SetOldSecret(const TString& secret) {
        OldSecret_ = secret;
    }

    void TClient::MarkDeleted() {
        IsDeleted_ = true;
    }

    bool TClient::IsDeleted() const {
        return IsDeleted_;
    }

    bool TClient::IsCorrect() const {
        return !Secret_.empty();
    }

    TClientId TClient::Id() const {
        return Id_;
    }

    std::optional<i64> TClient::GetAbcId() const {
        return AbcId_;
    }

    void TClient::SetAbcId(i64 id) {
        AbcId_ = id;
    }

    void TClient::SetName(const TString& name) {
        Name_ = name;
    }

    const TString& TClient::Name() const {
        return Name_;
    }
}
