#include "utils.h"

#include "client.h"

#include <passport/infra/daemons/tvmapi/src/output/result.h>
#include <passport/infra/daemons/tvmapi/src/output/serializer.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/dbfetcher.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/notify_log.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/runtime_context.h>
#include <passport/infra/daemons/tvmapi/src/runtime_context/strings.h>

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/handle.h>
#include <passport/infra/libs/cpp/dbpool/result.h>
#include <passport/infra/libs/cpp/dbpool/util.h>
#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <contrib/libs/openssl/include/openssl/evp.h>
#include <contrib/libs/openssl/include/openssl/hmac.h>
#include <contrib/libs/openssl/include/openssl/sha.h>

#include <library/cpp/http/simple/http_client.h>

#include <util/stream/file.h>
#include <util/stream/str.h>
#include <util/system/file.h>
#include <util/system/fs.h>

namespace NPassport::NTvm {
    const TString& TUtils::GetRequiredArg(const NCommon::TRequest& req, const TString& arg) {
        const TString& result = req.GetArg(arg);
        if (result.empty()) {
            throw TMissingArgException() << arg;
        }
        return result;
    }

    const TString& TUtils::GetRequiredUIntArg(const NCommon::TRequest& req, const TString& arg) {
        const TString& result = GetRequiredArg(req, arg);
        ui64 dummy = 0;
        if (!TryIntFromString<10>(result, dummy)) {
            throw TIncorrectArgException::MustBeNumber(arg);
        }
        return result;
    }

    const TString& TUtils::GetUIntArg(const NCommon::TRequest& req, const TString& arg) {
        const TString& result = req.GetArg(arg);

        ui64 dummy = 0;
        if (result && !TryIntFromString<10>(result, dummy)) {
            throw TIncorrectArgException::MustBeNumber(arg);
        }

        return result;
    }

    void TUtils::WriteJsonArrayToFile(const std::vector<TString>& v, const TString& filepath) {
        TString fileCache;
        size_t size = 10;
        for (const auto& str : v) {
            size += str.size();
        }
        fileCache.reserve(size);

        fileCache.push_back('[');
        for (const TString& str : v) {
            fileCache.append(str).push_back(',');
        }
        if (fileCache.back() == ',') {
            fileCache.pop_back();
        }
        fileCache.push_back(']');

        WriteFileViaTmp(filepath, fileCache);
    }

    void TUtils::WriteFileViaTmp(const TString& filename, TStringBuf body) {
        try {
            size_t pos = filename.rfind('/');
            if (pos == TString::npos) {
                throw yexception() << "File path is malformed: " << filename;
            }

            if (!NFs::MakeDirectoryRecursive(filename.substr(0, pos).c_str())) {
                throw yexception() << "Disk cache does not exist: " << filename;
            }

            const TString tmpDiskCache = filename + ".tmp";
            {
                TFileOutput fileOutput(tmpDiskCache.c_str());
                fileOutput << body;
            }
            if (!NFs::Rename(tmpDiskCache.c_str(), filename.c_str())) {
                throw yexception() << "Failed to move disk cache";
            }

            TLog::Info() << "File is written to disk: " << filename;
        } catch (const std::exception& e) {
            TLog::Error() << "Failed to write cache to disk: " << e.what();
        }
    }

    bool TUtils::FetchWithRetries(TKeepAliveHttpClient& client,
                                  const TString& url,
                                  NUnistat::TTimeStat& hgram,
                                  ui32 tries,
                                  const TString& token,
                                  TString& out,
                                  size_t& retries,
                                  TDuration& successRespTime) {
        const TKeepAliveHttpClient::THeaders hs = {
            {"Authorization", TString(token)},
        };

        while (tries--) {
            TStringStream s;

            TInstant start = TInstant::Now();
            try {
                auto code = client.DoGet(url, &s, hs);
                TDuration respTime = TInstant::Now() - start;
                hgram.Insert(respTime);

                if (code != 200) {
                    ++retries;
                    TLog::Warning() << "Response code is " << code
                                    << ". Took: " << respTime
                                    << ". Url: " << url << ". Response: " << s.Str();
                    continue;
                }

                successRespTime = respTime;
                out = s.Str();
                return true;
            } catch (const TSystemError& e) {
                TDuration respTime = TInstant::Now() - start;
                hgram.Insert(respTime);
                TLog::Debug() << "Url: " << url
                              << ". Took: " << respTime
                              << ". exception (probably it is timeout): " << e.what() << " : " << s.Str();
            } catch (const std::exception& e) {
                TDuration respTime = TInstant::Now() - start;
                hgram.Insert(respTime);
                TLog::Debug() << "Url: " << url
                              << ". Took: " << respTime
                              << ". exception: " << e.what() << " : " << s.Str();
            }
            ++retries;
        }

        return false;
    }

    bool TUtils::CheckTs(time_t ts, time_t allowedDiff) {
        time_t diff = std::time(nullptr) - ts;
        if (diff < 0) {
            diff = -diff;
        }
        return diff <= allowedDiff;
    }
}
