import argparse
import datetime
import json
import subprocess
import sys

import library.python.tskv as tskv
import requests


USAGE_LIMIT = 20
COUNTER_KEY = '_counter'
COUNTER_BY_HOUR_KEY = '_counter_by_hour'


class Emailer:
    def __init__(self, date, oauth_token, wet_run):
        self._date = "%s.%s.%s" % (date[:4], date[4:6], date[6:])
        self._oauth_token = oauth_token
        self._wet_run = wet_run
        self._emails = {}

    def flush(self):
        for login, text in self._emails.items():
            text = "Dear %s,\n\n%s\n" % (login, text)
            text += "If you have any questions, you can ask them here: https://st.yandex-team.ru/createTicket?queue=PASSPORTDUTY&_form=77618\n"
            subject = "Dear %s! Please fix warnings in TVM usage: %s" % (login, self._date)
            self.__send_emails(text, login, subject)

    def add(self, events):
        for _, d in events.get_deleted_src().items():
            self._add_src(d)
        for _, d in events.get_deleted_dst().items():
            self._add_dst(d)
        for _, d in events.get_old_secret().items():
            self._add_old_secret(d)

    def _add_src(self, d):
        count = d[COUNTER_KEY]
        if count < USAGE_LIMIT:
            return

        logins = self.__get_src_info(d)

        text = "Please stop using deleted tvm_id as SRC: %s\n" % self.__gen_src_description(d)
        text += "You used it %s time(s) during %s\n" % (count, self._date)

        hits = self._print_hits(d)

        self.__add_email(text, hits, logins)

    def _add_dst(self, d):
        count = d[COUNTER_KEY]
        if count < USAGE_LIMIT:
            return

        logins = self.__get_src_info(d)

        text = "Please stop using deleted tvm_id as DST: %s\n" % self.__gen_dst_description(d)
        text += "You used it %s time(s) during %s\n" % (count, self._date)
        text += "Your SRC was: %s\n" % self.__gen_src_description(d)

        hits = self._print_hits(d)

        self.__add_email(text, hits, logins)

    def _add_old_secret(self, d):
        logins = self.__get_src_info(d)

        text = "Please stop using old secret for tvm_id: %s\n" % self.__gen_src_description(d)
        text += "Probably you started secret rotation process (https://wiki.yandex-team.ru/passport/tvm2/client-secrets-renewal/)\n"
        text += "You can see rate of requests here (Golovan can miss several hits):\n"
        text += (
            "    https://yasm.yandex-team.ru/chart/signals=tvmconsumersoldsecret-%s_dmmm;hosts=CON;itype=passporttvmapi;ctype=production.intranet;geo=iva,man,myt,sas,vla;prj=passport.tvm;tier=none/?range=86400000\n"  # noqa
            % d['src']
        )
        text += "You used it %s time(s) during %s\n" % (d[COUNTER_KEY], self._date)

        hits = self._print_hits(d)

        self.__add_email(text, hits, logins)

    def __add_email(self, text, hits, logins):
        recipients_staff = tuple('https://staff.yandex-team.ru/' + login for login in logins)
        for i, login in enumerate(logins):
            e = self._emails.get(login, "")

            note = ""
            if len(logins) > 1:
                note += "NOTE: This report was also sent to:\n\t%s\n" % '\n\t'.join(
                    (staff for k, staff in enumerate(recipients_staff) if k != i)
                )
            e += text + note + hits + "\n"

            self._emails[login] = e

    def __send_emails(self, text, login, subject):
        if len(login) == 0:
            print('No login for email:\n%s' % text)
            return

        emails = ['cerevra@yandex-team.ru']  # for debug
        emails = ['tvm-notify@yandex-team.ru', '%s@yandex-team.ru' % login]

        if not self._wet_run:
            print('%s:\n%s\n\n' % (emails, text))
            return

        r = subprocess.run(
            ["mail", "-r", "noreply@yandex-team.ru", "-s", subject, "--"] + emails,
            input=text.encode('utf-8'),
            capture_output=True,
        )
        print(
            "sent email (%d bytes) to %s: code %s. stdout=%s. stderr=%s"
            % (len(text), emails, r.returncode, r.stdout, r.stderr)
        )

    def __gen_src_description(self, d):
        return self.__gen_description(d['src'], d['src_name'], d['src_abc'])

    def __gen_dst_description(self, d):
        return self.__gen_description(d['dst'], d['dst_name'], d['dst_abc'])

    def __gen_description(self, client_id, name, abcid):
        res = "%s (%s) - abc: " % (client_id, name)
        if abcid is None:
            res += 'not linked'
        else:
            res += 'https://abc.yandex-team.ru/services/%s' % abcid
        return res

    def __get_src_info(self, d):
        abc_id = d['src_abc']
        tvm_id = d['src']
        if tvm_id in ['2012028']:  # PASSP-35128
            abc_id = d['dst_abc']
            tvm_id = d['dst']

        res = []
        if abc_id is None:
            print('email was not sent: no abc info: tvm_id=%s' % tvm_id)
            return res

        try:
            r = requests.get(
                'https://abc-back.yandex-team.ru/api/v4/services/members/'
                + '?format=json&role__scope=tvm_management&service=%s' % abc_id,
                headers={'Authorization': 'OAuth ' + self._oauth_token},
            )
            if r.status_code != 200:
                print('abc_id %s was not found' % abc_id, file=sys.stderr)
                return
            response = json.loads(r.text)

            res = [result['person']['login'] for result in response['results']]
            res = list(set(res))
        except Exception as e:
            print('Visiting ABC: %s: %s' % (tvm_id, e), file=sys.stderr)
            raise

        return res

    def _print_hits(self, d):
        counters = [(k, v) for k, v in d[COUNTER_BY_HOUR_KEY].items()]
        counters.sort(key=lambda x: x[0])

        prepared = []
        for hour, ips in counters:
            hour = int(hour)
            prepared.append(
                (
                    hour,
                    {
                        'ip': ips,
                        'hour_end': hour,
                    },
                )
            )

        grouped = [prepared[0]]
        for idx in range(1, len(prepared)):
            last_grouped = grouped[-1]
            candidate = prepared[idx]
            if (
                last_grouped[1]['hour_end'] + 1 != candidate[0]
                or last_grouped[1]['ip'].keys() != candidate[1]['ip'].keys()
            ):
                grouped.append(candidate)
                continue

            last_grouped[1]['hour_end'] = candidate[0]
            for ip, count in candidate[1]['ip'].items():
                last_grouped[1]['ip'][ip] += count

        def print_hour(h):
            return str(h).zfill(2)

        res = 'SRC ip by hour:\n'
        for hour, ips in grouped:
            res += '%s:00-%s:59\n' % (print_hour(hour), print_hour(ips['hour_end']))

            ip_sorted = [(count, ip) for ip, count in ips['ip'].items()]
            ip_sorted.sort(key=lambda x: x[1])
            ip_sorted.sort(key=lambda x: x[0], reverse=True)

            for count, ip in ip_sorted:
                res += '%10s  %s\n' % (count, ip)

        return res


class Events:
    DELETED_SRC_KEYS = [
        'src',
        'src_name',
        'src_abc',
    ]

    DELETED_DST_KEYS = [
        'src',
        'src_name',
        'src_abc',
        'dst',
        'dst_name',
        'dst_abc',
    ]

    OLD_SECRET_KEYS = [
        'src',
        'src_name',
        'src_abc',
    ]

    def __init__(self):
        self._ignore = IgnoreList()
        self._deleted_src = {}
        self._deleted_dst = {}
        self._old_secret = {}
        self._handlers = {
            'deleted_src': self._add_deleted_src,
            'deleted_dst': self._add_deleted_dst,
            'old_secret': self._add_old_secret,
        }

    def add(self, d):
        t = d['type']

        h = self._handlers.get(t)
        if h is None:
            print('Skipping unknown type: %s' % t, file=sys.stderr)
        else:
            h(d)

    def is_ignored(self, d, id):
        return self._ignore.is_ignored(d[id], int(d['unixtime']))

    def _add_deleted_src(self, d):
        if self.is_ignored(d, 'src'):
            return

        data = self._deleted_src.setdefault(d['src'], dict())
        self._proc_default_keys(d, Events.DELETED_SRC_KEYS, data)

    def _add_deleted_dst(self, d):
        if self.is_ignored(d, 'dst'):
            return

        data = self._deleted_dst.setdefault('%s_%s' % (d['src'], d['dst']), dict())
        self._proc_default_keys(d, Events.DELETED_DST_KEYS, data)

    def _add_old_secret(self, d):
        if self.is_ignored(d, 'src'):
            return

        data = self._old_secret.setdefault(d['src'], dict())
        self._proc_default_keys(d, Events.OLD_SECRET_KEYS, data)

    def _proc_default_keys(self, d, keyset, out):
        for key in keyset:
            out[key] = d[key]

        if COUNTER_KEY in out:
            out[COUNTER_KEY] += 1
        else:
            out[COUNTER_KEY] = 1

        cbh = out.setdefault(COUNTER_BY_HOUR_KEY, dict())
        h = d['datetime'].split('T')[1].split(':')[0]
        iph = cbh.setdefault(h, dict())
        ip = d['src_ip']

        if ip in iph:
            iph[ip] += 1
        else:
            iph[ip] = 1

    def get_deleted_src(self):
        return self._deleted_src

    def get_deleted_dst(self):
        return self._deleted_dst

    def get_old_secret(self):
        return self._old_secret


class IgnoreList:
    IGNORED_TVMID_TO_MAXUNIXTIME = {
        "2000134": 1607669134,
    }

    def is_ignored(self, tvm_id, unixtime):
        up_to = IgnoreList.IGNORED_TVMID_TO_MAXUNIXTIME.get(tvm_id)
        return up_to is not None and unixtime < up_to


def parse_log(filename):
    res = Events()

    for line in open(filename):
        try:
            line = line[:-1].split('tskv\t')[1]

            d = {}
            for k, v in tskv.loads(line).items():
                value = v.decode('utf-8')
                if len(value) == 0:
                    value = None
                d[k.decode('utf-8')] = value

            assert d['tskv_format'] == 'tvm-notify'

            res.add(d)
        except Exception as e:
            print('failed to parse line: "%s": %s' % (line, e))
            # raise

    return res


def run(argv):
    parser = argparse.ArgumentParser(description='send notifications (emails) to TVM-managers')
    parser.add_argument('--filename', type=str, help='filename with log to parse')
    parser.add_argument('--wet_run', action='store_true', help='disable dry run')
    parser.add_argument('--oauth_token', type=str, help='oauth token to use ABC')
    args = parser.parse_args(args=argv)

    filename = (
        args.filename
        if args.filename is not None
        else (datetime.datetime.now() - datetime.timedelta(days=1)).strftime(
            '/storage/logs/production-intranet/var/log/fastcgi2/tvm-notify.log.%Y%m%d'
        )
    )
    print(filename + '\n')

    events = parse_log(filename)

    emailer = Emailer(date=filename[-8:], oauth_token=args.oauth_token, wet_run=args.wet_run)
    emailer.add(events)
    emailer.flush()
