#include "stats.h"

#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/http/simple/http_client.h>
#include <library/cpp/string_utils/tskv_format/tskv_map.h>
#include <library/cpp/tvmauth/client/facade.h>

#include <util/stream/file.h>
#include <util/stream/format.h>
#include <util/stream/str.h>
#include <util/string/builder.h>

#include <vector>

namespace NPassport::NParseAccess {
    TTvmConfig TTvmConfig::ReadFromFile(const TString& filename) {
        const TString body = TFileInput(filename).ReadAll();

        rapidjson::Document doc;
        Y_ENSURE(NJson::TReader::DocumentAsObject(body, doc));

        TTvmConfig res;

        Y_ENSURE(NJson::TReader::MemberAsUInt(doc, "tvm_id", res.TvmId));
        Y_ENSURE(NJson::TReader::MemberAsString(doc, "tvm_secret", res.Secret));

        Y_ENSURE(res.TvmId);
        Y_ENSURE(res.Secret);

        return res;
    }

    static const ui32 ABC_TVM_ID = 2012190;

    static TString MakeRate(ui64 part, ui64 total) {
        if (part == 0 || total == 0) {
            return "0";
        }

        TStringStream s;
        s << Prec(100. * part / total, PREC_POINT_DIGITS, 1) << "% ";
        s << "("
          << Prec(part / 86400., PREC_POINT_DIGITS, 1) << " rps - "
          << part << "/" << total
          << ")";
        return s.Str();
    }

    static TString SerializeSet(const THashSet<TString>& set) {
        std::vector<TString> vec(set.begin(), set.end());
        std::sort(vec.begin(), vec.end());

        TStringStream s;
        for (const TString& v : vec) {
            s << v << Endl;
        }
        return NUtils::BinToBase64(s.Str());
    }

    TAllClientInfo::TAllClientInfo()
        : Dummy_({.Name = "NULL"})
    {
    }

    void TAllClientInfo::Reserve(size_t size) {
        Data_.reserve(size);
    }

    void TAllClientInfo::Add(const TString& clientId) {
        Data_.emplace(clientId, std::optional<TClientInfo>());
    }

    void TAllClientInfo::Add(const TString& clientId, TClientInfo&& info) {
        Data_.emplace(clientId, std::move(info));
    }

    void TAllClientInfo::SetTags(const TString& clientId, THashSet<TString>&& tags) {
        auto it = Data_.find(clientId);
        if (it == Data_.end()) {
            it = Data_.emplace(clientId, Dummy_).first;
        }

        if (!it->second) {
            it->second = Dummy_;
        }

        it->second->AbcTags = std::move(tags);
    }

    const TClientInfo& TAllClientInfo::Get(const TString& clientId) const {
        auto it = Data_.find(clientId);
        if (it == Data_.end()) {
            return Dummy_;
        }

        return it->second ? *it->second : Dummy_;
    }

    void TTvm2::Add(ui32 code, TDuration respTime, TStringBuf path) {
        ++TotalCount_;
        TotalTime_ += respTime;

        MaxRespTime_ = std::max(MaxRespTime_, respTime);

        if (code == 400) {
            ++Code400Count_;
        }

        auto it = Pathes_.find(path);
        if (it == Pathes_.end()) {
            it = Pathes_.emplace(TString(path), 0).first;
        }
        ++it->second;
    }

    void TTvm2::AddGrantType(const TRequest& req, TStringBuf grantType, TStringBuf clientId) {
        TConsumers& consumers = GrantTypes_.emplace(grantType, TConsumers()).first->second;
        TConsumerStats& consumer = consumers.emplace(clientId, TConsumerStats()).first->second;

        ++consumer.TotalCount;
        consumer.AllReqsTime + req.ResponseTime;

        if (req.HttpCode == 400) {
            ++consumer.Code400Count;
        }
        if (req.IsHttps) {
            ++consumer.Https;
        }

        consumer.Ips.emplace(req.Ip);

        for (const auto& pair : req.Params) {
            consumer.Args.emplace(pair.first);
        }
    }

    void TTvm2::AddDsts(const TRequest& req, TStringBuf grantType, TStringBuf clientId) {
        TConsumers& consumers = GrantTypes_.emplace(grantType, TConsumers()).first->second;
        TConsumerStats& consumer = consumers.emplace(clientId, TConsumerStats()).first->second;

        TStringBuf dsts;
        if (auto it = req.Params.find("dst"); it != req.Params.end()) {
            dsts = it->second;
        }
        while (dsts) {
            consumer.Dsts.emplace(dsts.NextTok(','));
        }
    }

    void TTvm2::Print(const TAllClientInfo& clientInfo, IOutputStream& output) const {
        PrintCommon(output);
        output << Endl;
        PrintPathes(output);
        output << Endl;
        PrintComsumers(clientInfo, output);
        output << Endl;
    }

    void TTvm2::PrintTskv(const TAllClientInfo& clientInfo, IOutputStream& output) const {
        for (const auto& pair_granttype : GrantTypes_) {
            for (const auto& pair_clientid : pair_granttype.second) {
                const TString& clientId = pair_clientid.first;
                const TConsumerStats& cons = pair_clientid.second;

                TMap<TString, TString> map{
                    {"tvm_id", pair_clientid.first},
                    {"tvm_name", clientInfo.Get(clientId).Name},
                    {"abc_id", IntToString<10>(clientInfo.Get(clientId).AbcServiceId)},
                    {"grant_type", pair_granttype.first},
                    {"total_reqs", IntToString<10>(cons.TotalCount)},
                    {"invalid_reqs", IntToString<10>(cons.Code400Count)},
                    {"avg_resp_time", TStringBuilder() << (cons.AllReqsTime.MilliSeconds() / cons.TotalCount / 1000.)},
                    {"https", IntToString<10>(cons.Https)},
                    {"all_ip", SerializeSet(cons.Ips)},
                    {"args", SerializeSet(cons.Args)},
                    {"dsts", SerializeSet(cons.Dsts)},
                    {"abc_tags", SerializeSet(clientInfo.Get(clientId).AbcTags)},
                };

                TString s;
                output << NTskvFormat::SerializeMap(map, s) << Endl;
            }
        }
    }

    void TTvm2::PrintCommon(IOutputStream& output) const {
        output << "Code 400: " << MakeRate(Code400Count_, TotalCount_) << Endl;
        output << "Average response time: " << AvgTime() << Endl;
        output << "Maximum response time: " << MaxRespTime_ << Endl;
    }

    void TTvm2::PrintPathes(IOutputStream& output) const {
        std::vector<std::pair<TString, ui64>> pathes(Pathes_.begin(), Pathes_.end());
        std::sort(pathes.begin(), pathes.end(), [](const auto& l, const auto& r) {
            return l.second > r.second;
        });
        for (const auto& pair : pathes) {
            output << pair.first << " -> " << MakeRate(pair.second, TotalCount_) << Endl;
        }
    }

    void TTvm2::PrintComsumers(const TAllClientInfo& clientInfo, IOutputStream& output) const {
        for (const auto& pair : GrantTypes_) {
            ui64 total = 0;
            ui64 total400 = 0;
            for (const auto& p : pair.second) {
                total += p.second.TotalCount;
                total400 += p.second.Code400Count;
            }

            output << "---------- grant_type=" << pair.first
                   << ": " << MakeRate(total, TotalCount_) << Endl;

            std::vector<std::pair<TString, TConsumerStats>> cons(pair.second.begin(), pair.second.end());
            std::sort(cons.begin(), cons.end(), [](const auto& l, const auto& r) {
                return l.second.TotalCount > r.second.TotalCount;
            });
            for (const auto& p : cons) {
                output << LeftPad(p.first, 13) << " ("
                       << LeftPad(clientInfo.Get(p.first).Name, 50)
                       << ") "
                       << MakeRate(p.second.TotalCount, total)
                       << " -> 400: "
                       << MakeRate(p.second.Code400Count, total400)
                       << Endl;
            }
        }
    }

    void TStats::Add(TRequest&& req) {
        ++Common_.TotalCount;
        if (req.Path == "/ping" || req.Path.Contains("/nagios")) {
            ++Common_.PingCount;
            return;
        }

        AddTvm2(std::move(req));
    }

    void TStats::FetchInfo(const TTvmConfig& tvm) {
        FetchClientInfo();
        FetchAbcTags(tvm);
    }

    void TStats::Print(IOutputStream& output) const {
        output << "/ping: " << MakeRate(Common_.PingCount, Common_.TotalCount) << Endl;
        output << Endl;

        output << "======> TVM 2.0: " << MakeRate(Tvm2_.TotalCount(), Common_.TotalCount) << Endl;
        Tvm2_.Print(ClientInfo_, output);
        output << Endl;
    }

    void TStats::PrintTskv(IOutputStream& output) const {
        Tvm2_.PrintTskv(ClientInfo_, output);
    }

    void TStats::AddTvm2(TRequest&& req) {
        Tvm2_.Add(req.HttpCode, req.ResponseTime, req.Path);

        auto itSrc = req.Params.find("src");

        if (req.Path == "/2/ticket") {
            auto itGrantType = req.Params.find("grant_type");
            if (itGrantType != req.Params.end() && itSrc != req.Params.end()) {
                AllClientIds_.emplace(itSrc->second);

                Tvm2_.AddGrantType(req, itGrantType->second, itSrc->second);
                Tvm2_.AddDsts(req, itGrantType->second, itSrc->second);
            }
        }
    }

    void TStats::FetchClientInfo() {
        ClientInfo_.Reserve(AllClientIds_.size());

        TKeepAliveHttpClient http("https://tvm.yandex-team.ru", 443);

        for (const TString& clientId : AllClientIds_) {
            ui32 tries = 5;
            const TString url = NUtils::CreateStr("/client/", clientId, "/info");

            for (ui32 idx = 0; idx < tries; ++idx) {
                TStringStream output;
                try {
                    auto code = http.DoGet(url, &output);
                    Y_ENSURE(code == 200 || code == 404, "Unexpected code : " << code);

                    if (code == 200) {
                        rapidjson::Document doc;
                        Y_ENSURE(NJson::TReader::DocumentAsObject(output.Str(), doc), "Invalid json");

                        TClientInfo info;
                        Y_ENSURE(NJson::TReader::MemberAsString(doc, "name", info.Name), "Missing 'name'");
                        NJson::TReader::MemberAsUInt64(doc, "abc_service_id", info.AbcServiceId);

                        ClientInfo_.Add(clientId, std::move(info));
                    } else {
                        ClientInfo_.Add(clientId);
                    }

                    break;
                } catch (const std::exception& e) {
                    Cerr << "Failed to get client info: " << clientId
                         << ": " << e.what()
                         << ". Output: " << output.Str()
                         << Endl;
                }
            }
        }
    }

    static const TString ABC = "Abc";
    static const TString X_YA_SERVICE_TICKET = "X-Ya-Service-Ticket";

    static NTvmAuth::TTvmClient CreateTvmClient(const TTvmConfig& tvmCfg) {
        using namespace NTvmAuth;

        NTvmApi::TClientSettings settings;
        settings.SetSelfTvmId(tvmCfg.TvmId);
        settings.EnableServiceTicketsFetchOptions(
            tvmCfg.Secret,
            NTvmApi::TClientSettings::TDstMap{
                {ABC, ABC_TVM_ID},
            });
        return TTvmClient(settings, MakeIntrusive<TCerrLogger>(7));
    }

    void TStats::FetchAbcTags(const TTvmConfig& tvmCfg) {
        const size_t pageSize = 500;
        std::vector<TString> chunks;
        chunks.reserve(AllClientIds_.size() / pageSize + 1);
        chunks.push_back({});

        size_t lastChunkSize = 0;
        for (const TString& id : AllClientIds_) {
            if (lastChunkSize >= pageSize) {
                chunks.push_back({});
                chunks.back().reserve(pageSize * 8);
                lastChunkSize = 0;
            }
            if (!chunks.back().empty()) {
                chunks.back().append(",");
            }
            chunks.back().append(id);
            ++lastChunkSize;
        }

        const NTvmAuth::TTvmClient tvm = CreateTvmClient(tvmCfg);
        TKeepAliveHttpClient http("https://abc-back.yandex-team.ru", 443);

        for (const TString& chunk : chunks) {
            const TString url = NUtils::CreateStr(
                "/api/v4/resources/consumers/",
                "?supplier=14",
                "&type=47",
                "&page_size=", pageSize,
                "&fields=tags.slug,resource.external_id",
                "&resource__external_id__in=", chunk);

            ui32 tries = 5;
            for (size_t idx = 0; idx < tries; ++idx) {
                TStringStream output;

                try {
                    auto code = http.DoGet(url, &output, {{X_YA_SERVICE_TICKET, tvm.GetServiceTicketFor(ABC)}});
                    Y_ENSURE(code == 200, "Bad code: " << code);

                    rapidjson::Document doc;
                    Y_ENSURE(NJson::TReader::DocumentAsObject(output.Str(), doc), "Invalid json");
                    const rapidjson::Value* results = nullptr;
                    Y_ENSURE(NJson::TReader::MemberAsArray(doc, "results", results), "Missing 'results'");

                    for (size_t idx = 0; idx < results->Size(); ++idx) {
                        const rapidjson::Value& result = (*results)[idx];
                        const rapidjson::Value* resource = nullptr;
                        Y_ENSURE(NJson::TReader::MemberAsObject(result, "resource", resource), "Missing 'resource'");

                        TString clientId;
                        Y_ENSURE(NJson::TReader::MemberAsString(*resource, "external_id", clientId), "Missing 'external_id'");

                        const rapidjson::Value* tags = nullptr;
                        Y_ENSURE(NJson::TReader::MemberAsArray(result, "tags", tags), "Missing 'tags'");

                        THashSet<TString> res;
                        for (size_t idx = 0; idx < tags->Size(); ++idx) {
                            TString val;
                            Y_ENSURE(NJson::TReader::MemberAsString((*tags)[idx], "slug", val), "Missing 'slug'");
                            res.insert(val);
                        }

                        ClientInfo_.SetTags(clientId, std::move(res));
                    }

                } catch (const std::exception& e) {
                    Cerr << "Failed to get abc tags: "
                         << ": " << e.what()
                         << ". Output: " << output.Str()
                         << Endl;
                }
            }
        }
    }
}
