#pragma once

#include <util/datetime/base.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/map.h>
#include <util/generic/set.h>
#include <util/generic/string.h>

namespace NPassport::NParseAccess {
    struct TTvmConfig {
        ui32 TvmId = 0;
        TString Secret;

        static TTvmConfig ReadFromFile(const TString& filename);
    };

    using TParams = THashMap<TStringBuf, TStringBuf>;

    struct TRequest {
        bool IsHttps = false;
        TStringBuf Ip;
        TDuration ResponseTime;
        ui32 HttpCode = 0;
        TStringBuf Path;
        TParams Params;
    };

    struct TConsumerStats { // NOLINT(bugprone-exception-escape)
        ui64 TotalCount = 0;
        ui64 Code400Count = 0;
        TDuration AllReqsTime;
        ui64 Https = 0;
        THashSet<TString> Ips;
        THashSet<TString> Args;
        THashSet<TString> Dsts;
    };

    struct TClientInfo { // NOLINT(bugprone-exception-escape)
        TString Name;
        ui64 AbcServiceId = 0;
        THashSet<TString> AbcTags;
    };

    class TAllClientInfo { // NOLINT(bugprone-exception-escape)
    public:
        TAllClientInfo();
        void Reserve(size_t size);

        void Add(const TString& clientId);
        void Add(const TString& clientId, TClientInfo&& info);

        void SetTags(const TString& clientId, THashSet<TString>&& tags);

        const TClientInfo& Get(const TString& clientId) const;

    private:
        THashMap<TString, std::optional<TClientInfo>> Data_;
        const TClientInfo Dummy_;
    };

    class TTvm2 { // NOLINT(bugprone-exception-escape)
        using TConsumers = THashMap<TString, TConsumerStats>;
        using TGrantTypes = TMap<TString, TConsumers>;

    public:
        void Add(ui32 code, TDuration respTime, TStringBuf path);

        void AddGrantType(const TRequest& req, TStringBuf grantType, TStringBuf clientId);
        void AddDsts(const TRequest& req, TStringBuf grantType, TStringBuf clientId);

        void Print(const TAllClientInfo& clientInfo, IOutputStream& output) const;
        void PrintTskv(const TAllClientInfo& clientInfo, IOutputStream& output) const;

        ui64 TotalCount() const {
            return TotalCount_;
        }

        TDuration AvgTime() const {
            if (TotalCount_ == 0) {
                return {};
            }

            return TotalTime_ / TotalCount_;
        }

        TDuration MaxTime() const {
            return MaxRespTime_;
        }

        ui64 Code400() const {
            return Code400Count_;
        }

    private:
        void PrintCommon(IOutputStream& output) const;
        void PrintPathes(IOutputStream& output) const;
        void PrintComsumers(const TAllClientInfo& clientInfo, IOutputStream& output) const;

    private:
        ui64 TotalCount_ = 0;
        TDuration TotalTime_;
        ui64 Code400Count_ = 0;
        TDuration MaxRespTime_;

        THashMap<TString, ui64> Pathes_;
        TGrantTypes GrantTypes_;
    };

    class TStats { // NOLINT(bugprone-exception-escape)
    public:
        struct TCommon {
            ui64 PingCount = 0;
            ui64 TotalCount = 0;
        };

        void Add(TRequest&& req);
        void FetchInfo(const TTvmConfig& tvm);

        void Print(IOutputStream& output) const;
        void PrintTskv(IOutputStream& output) const;

    private:
        void AddTvm2(TRequest&& req);

        void FetchClientInfo();
        void FetchAbcTags(const TTvmConfig& tvmCfg);

    private:
        TTvm2 Tvm2_;
        TCommon Common_;

        THashSet<TString> AllClientIds_;
        TAllClientInfo ClientInfo_;
    };
}
