#include <passport/infra/daemons/tvmapi/src/runtime_context/dbfetcher.h>
#include <passport/infra/daemons/tvmapi/src/utils/client.h>

#include <passport/infra/libs/cpp/json/writer.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/tvmauth/src/rw/keys.h>

using namespace NPassport;
using namespace NPassport::NTvm;

Y_UNIT_TEST_SUITE(DbFetcher) {
    Y_UNIT_TEST(fillClientSecret) {
        TClient cl(1);
        NTvmCommon::TDecryptor decr("BtniSXCXhroPtOgXA61i5ZxSeX/solWR");

        TDbCacheBuilder::FillClientSecret(cl,
                                          "2:AJ1bRQJVdj50UoTB:jP4Hj4gBj40LRo6tnnN1Fy34VNLo3by4NJjTX9KL:Uun5pI3hMBzEMcODjKA1fQ",
                                          decr);
        UNIT_ASSERT_STRINGS_EQUAL("7YeZvmiGBJMeEzdu-qxGTg",
                                  NUtils::Bin2base64url(cl.Secret()));

        cl = TClient(0);
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TDbCacheBuilder::FillClientSecret(cl,
                                              "2:AJ1bRQJVdj50UoTB:jP4Hj4gBj40LRo6tnnN1Fy34VNLo3by4NJjTX9KL:Uun5pI3hMBzEMcODjKA1fQ",
                                              decr),
            yexception,
            "failed to decrypt secret");
        UNIT_ASSERT_STRINGS_EQUAL("", cl.Secret());
    }

    Y_UNIT_TEST(fillKeyIds) {
        TClient cl(0);
        TDbCacheBuilder::FillKeyIds(cl, "");
        UNIT_ASSERT(cl.KeysIds().empty());

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TDbCacheBuilder::FillKeyIds(cl, ",123,234,"),
            yexception,
            "invalid key id: ',123,234,'");
        UNIT_ASSERT(cl.KeysIds().empty());

        TDbCacheBuilder::FillKeyIds(cl, "|123|234|");
        UNIT_ASSERT_EQUAL(std::vector<TKeyId>({123, 234}), cl.KeysIds());
    }

    Y_UNIT_TEST(fillClientOldSecret) {
        NTvmCommon::TDecryptor decr("BtniSXCXhroPtOgXA61i5ZxSeX/solWR");

        {
            TClient cl(0);
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                TDbCacheBuilder::FillClientOldSecret(
                    cl,
                    "2:AJ1bRQJVdj50UoTB:jP4Hj4gBj40LRo6tnnN1Fy34VNLo3by4NJjTX9KL:Uun5pI3hMBzEMcODjKA1fQ",
                    decr),
                yexception,
                "failed to decrypt old secret");
        }

        TClient cl(1);
        TDbCacheBuilder::FillClientOldSecret(cl,
                                             "2:AJ1bRQJVdj50UoTB:jP4Hj4gBj40LRo6tnnN1Fy34VNLo3by4NJjTX9KL:Uun5pI3hMBzEMcODjKA1fQ",
                                             decr);
        UNIT_ASSERT_STRINGS_EQUAL("7YeZvmiGBJMeEzdu-qxGTg", NUtils::Bin2base64url(cl.OldSecret()));
    }

    Y_UNIT_TEST(getBinaryKey) {
        NTvmCommon::TDecryptor decr("BtniSXCXhroPtOgXA61i5ZxSeX/solWR");

        UNIT_ASSERT_STRINGS_EQUAL("7YeZvmiGBJMeEzdu-qxGTg",
                                  NUtils::Bin2base64url(TDbCacheBuilder::GetBinaryKey(
                                      "2:AJ1bRQJVdj50UoTB:jP4Hj4gBj40LRo6tnnN1Fy34VNLo3by4NJjTX9KL:Uun5pI3hMBzEMcODjKA1fQ",
                                      1,
                                      decr)));

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TDbCacheBuilder::GetBinaryKey(
                "2:AJ1bRQJVdj50UoTB:jP4Hj4gBj40LRo6tnnN1Fy34VNLo3by4NJjTX9KL:Uun5pI3hMBzEMcODjKA1fQ",
                0,
                decr),
            yexception,
            "failed to decrypt");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TDbCacheBuilder::GetBinaryKey(
                "2:AJ1bRQJVdj50UoTB:jP4Hj4gBj40LRo6tnnN1Fy34VNLo3by4NJjT:Uun5pI3hMBzEMcODjKA1fQ",
                1,
                decr),
            yexception,
            "failed to decrypt");
    }

    Y_UNIT_TEST(addClientAttr) {
        NTvmCommon::TDecryptor decr("BtniSXCXhroPtOgXA61i5ZxSeX/solWR");

        NUnistat::TSignalDiff<> errors("");
        TDbCacheBuilder b(decr, errors);
        b.AddClientAttr(1, 3, "|1100000000000007|");
        b.AddClientAttr(1, 5, "2:AJ1bRQJVdj50UoTB:jP4Hj4gBj40LRo6tnnN1Fy34VNLo3by4NJjTX9KL:Uun5pI3hMBzEMcODjKA1fQ");
        b.AddClientAttr(1, 6, "|1|");
        b.AddClientAttr(1, 7, "|client_credentials|");
        b.AddClientAttr(1, 9, "67");
        b.PostProcess();
        TDbCache c = b.Finalize();
        UNIT_ASSERT_EQUAL(1, c.Clients->size());
    }

    Y_UNIT_TEST(addClientAttr_deleted) {
        NTvmCommon::TDecryptor decr("BtniSXCXhroPtOgXA61i5ZxSeX/solWR");

        NUnistat::TSignalDiff<> errors("");
        TDbCacheBuilder b(decr, errors);
        b.AddClientAttr(1, 3, "|1100000000000007|");
        b.AddClientAttr(1, 5, "2:AJ1bRQJVdj50UoTB:jP4Hj4gBj40LRo6tnnN1Fy34VNLo3by4NJjTX9KL:Uun5pI3hMBzEMcODjKA1fQ");
        b.AddClientAttr(1, 6, "|1|");
        b.AddClientAttr(1, 7, "|client_credentials|");
        b.AddClientAttr(1, 9, "67");
        b.AddClientAttr(1, 18, "1");
        b.PostProcess();
        TDbCache c = b.Finalize();
        UNIT_ASSERT_EQUAL(1, c.Clients->size());
        UNIT_ASSERT(c.Clients->begin()->second->IsDeleted());
    }

    Y_UNIT_TEST(addKeyAttr) {
        NTvmCommon::TDecryptor decr("BtniSXCXhroPtOgXA61i5ZxSeX/solWR");

        NUnistat::TSignalDiff<> errors("");
        TDbCacheBuilder b(decr, errors);
        b.AddClientAttr(1, 3, "|1100000000000007|");
        b.AddClientAttr(1, 5, "2:AJ1bRQJVdj50UoTB:jP4Hj4gBj40LRo6tnnN1Fy34VNLo3by4NJjTX9KL:Uun5pI3hMBzEMcODjKA1fQ");
        b.AddClientAttr(1, 6, "|1|");
        b.AddClientAttr(1, 7, "|client_credentials|");
        b.AddClientAttr(1, 9, "67");
        b.AddKeyAttr(1, 1, "2:KMFgCFwyto7l_ntq:r_RuEIToSTxXHeC_DS7aTGDzgBXsz-fQ2Y_u9H7nPo1K-rhi05KiEm38bixymgqYlPjD6x2kohC3wTVy5i6w3hBjlpljdMiRJhGZ3mGwWizmwtW-hWgqOtvngI3WVA6LOV28BMCA9aX9obAhoMfp_rYERI3VZ1cPWvzPpXY0Ymbg3R-3y5nhmQ1kfTngq3dmDUlkZjbugK1Lxms753RIbXWQnMTWBnvY647CUNHnzUrlGo5dhoEdZQANlQ:k1WejHp6-iRhfOfh2VPXHw");
        b.AddKeyAttr(1, 2, "2:e9AAA489YJx1RF49:xc7kNvaauf28QcbT07hkM195jKttm1_w33whxtQwQ0lnOrn8BMH_UN21xwooCYjlbooDSuKQnjy6nRBSupIoSqbm42ULJs6gZr7MD31b4w79_zVWretDO4hQIvty-ZoOljt3KgJtDwJ6RhOsq39tJRsAQx1ih75bOIEpm_uk0K4PE5jNRmakcM411PhEkDOGf6LxV08PK6r2Nu3YVXHEvX7yoZdbZswtxFytaf37EB39rU6FKAEZUvvloE7Ne8xzuqFZBuwB4u9QfhThTrcKpl8Pz7EDrGci35suEBZyKiOmDH339LcGxOIpb2KvumW_A3a60X_k2uPoFEzU0O6XNLqLAD6_8Nj4gF_SkWjkpeG6NCbJV_fksU7JRcO6LASD3qMwLYHHMYYGC06VpAjLZs-j_60VXdm_5N440SbuoD9xnVHwg3TRyj1LX4B4hqAYHKrzMwPoL6WH0zcbVhVbN3U2LpNJGZoj4oE58zIs50_khezBDmyozcH-7mC8ngoVp9VR18MAb5Zm18SevYtzIoKTIWb15PdHuAXybOxZ5-DJdqE89eAIocIXlN1vwIqUaKkaUEb3_PZIINX_l-A9pLTRhuuWkXx2z5vaF3fSoTDB-KeSQx8XPo3rX9Suv9gzO14f4i4i3mlBD0tUwevZYfa2r18aE2l8HDbyNmN_zkWMVe_pDAKgz6VTItWJBTFMTHzZjowSMoqxDOOaloW3FR6HJNepIjmsYwdX6OxzqouQrFHKqAPdXM6AVHSAOkyMd2RPqnQynitWCADoI9WBRt8EcYHuyt4785KlYyeyLbVy1bYakYeXZBPtaFdmyfI7OUU5HAbbvEUFHRKUCdNIKJlfMuZmxxIm_OmkKYYy8mgxenRGvhJawv4zO78TIHAHCpykx4-N85Ht6iXmALIpbWqyzospUZWUXCJbQMTP6PeCTy4JWTyyvVy9QYrc4vS3BMEu7hVNfTsxNuIN1Btp5hSRyFYTPo8C6Z_zb4t_zllK0aClOkUPPu9uOaw7l0PKvtpGwb0WbJygV-p4EYKu0GRD7YKaZrxLWgK-_vcEB3Ec1V-t5WZR2w:hae8RDaIXw1cvuxAHLO0AQ");
        b.AddKeyAttr(1, 3, "1472218045");
        b.PostProcess();
        TDbCache c = b.Finalize();
        UNIT_ASSERT_EQUAL(1, c.Clients->size());
    }

    Y_UNIT_TEST(addKeyAttr_deleted) {
        NTvmCommon::TDecryptor decr("BtniSXCXhroPtOgXA61i5ZxSeX/solWR");

        NUnistat::TSignalDiff<> errors("");
        TDbCacheBuilder b(decr, errors);
        b.AddKeyAttr(1, 1, "2:KMFgCFwyto7l_ntq:r_RuEIToSTxXHeC_DS7aTGDzgBXsz-fQ2Y_u9H7nPo1K-rhi05KiEm38bixymgqYlPjD6x2kohC3wTVy5i6w3hBjlpljdMiRJhGZ3mGwWizmwtW-hWgqOtvngI3WVA6LOV28BMCA9aX9obAhoMfp_rYERI3VZ1cPWvzPpXY0Ymbg3R-3y5nhmQ1kfTngq3dmDUlkZjbugK1Lxms753RIbXWQnMTWBnvY647CUNHnzUrlGo5dhoEdZQANlQ:k1WejHp6-iRhfOfh2VPXHw");
        b.AddKeyAttr(1, 2, "2:e9AAA489YJx1RF49:xc7kNvaauf28QcbT07hkM195jKttm1_w33whxtQwQ0lnOrn8BMH_UN21xwooCYjlbooDSuKQnjy6nRBSupIoSqbm42ULJs6gZr7MD31b4w79_zVWretDO4hQIvty-ZoOljt3KgJtDwJ6RhOsq39tJRsAQx1ih75bOIEpm_uk0K4PE5jNRmakcM411PhEkDOGf6LxV08PK6r2Nu3YVXHEvX7yoZdbZswtxFytaf37EB39rU6FKAEZUvvloE7Ne8xzuqFZBuwB4u9QfhThTrcKpl8Pz7EDrGci35suEBZyKiOmDH339LcGxOIpb2KvumW_A3a60X_k2uPoFEzU0O6XNLqLAD6_8Nj4gF_SkWjkpeG6NCbJV_fksU7JRcO6LASD3qMwLYHHMYYGC06VpAjLZs-j_60VXdm_5N440SbuoD9xnVHwg3TRyj1LX4B4hqAYHKrzMwPoL6WH0zcbVhVbN3U2LpNJGZoj4oE58zIs50_khezBDmyozcH-7mC8ngoVp9VR18MAb5Zm18SevYtzIoKTIWb15PdHuAXybOxZ5-DJdqE89eAIocIXlN1vwIqUaKkaUEb3_PZIINX_l-A9pLTRhuuWkXx2z5vaF3fSoTDB-KeSQx8XPo3rX9Suv9gzO14f4i4i3mlBD0tUwevZYfa2r18aE2l8HDbyNmN_zkWMVe_pDAKgz6VTItWJBTFMTHzZjowSMoqxDOOaloW3FR6HJNepIjmsYwdX6OxzqouQrFHKqAPdXM6AVHSAOkyMd2RPqnQynitWCADoI9WBRt8EcYHuyt4785KlYyeyLbVy1bYakYeXZBPtaFdmyfI7OUU5HAbbvEUFHRKUCdNIKJlfMuZmxxIm_OmkKYYy8mgxenRGvhJawv4zO78TIHAHCpykx4-N85Ht6iXmALIpbWqyzospUZWUXCJbQMTP6PeCTy4JWTyyvVy9QYrc4vS3BMEu7hVNfTsxNuIN1Btp5hSRyFYTPo8C6Z_zb4t_zllK0aClOkUPPu9uOaw7l0PKvtpGwb0WbJygV-p4EYKu0GRD7YKaZrxLWgK-_vcEB3Ec1V-t5WZR2w:hae8RDaIXw1cvuxAHLO0AQ");
        b.AddKeyAttr(1, 3, "1472218045");
        b.AddKeyAttr(1, 4, "1");
        b.PostProcess();
        UNIT_ASSERT_EQUAL(1, b.Keys().size());
    }

    Y_UNIT_TEST(setTvmPrivateKey) {
        NTvmCommon::TDecryptor decr("BtniSXCXhroPtOgXA61i5ZxSeX/solWR");

        NUnistat::TSignalDiff<> errors("");
        TDbCacheBuilder b(decr, errors);
        b.AddClientAttr(1, 3, "|1100000000000007|");
        b.AddClientAttr(1, 5, "2:AJ1bRQJVdj50UoTB:jP4Hj4gBj40LRo6tnnN1Fy34VNLo3by4NJjTX9KL:Uun5pI3hMBzEMcODjKA1fQ");
        b.AddClientAttr(1, 6, "|143|");
        b.AddClientAttr(1, 7, "|client_credentials|");
        b.AddClientAttr(1, 9, "67");
        b.AddKeyAttr(143, 1, "2:MR9a9GrN92P0XvS4:DH4zVf78vGADOTSX-rh-0Hq2qU-bA3MK2cjuERf_lVGkMoQ9-paLakMmvBJcKN0n7Xg3Mnrh1XOKOh60Lwp6LidYos0saXxpO3rTYm7qHsFxAHazKQPYX43ODUIwjekyyhyQJN47WCX0lL0kLQxAkaSsjbHMRcgCUxbqHnXX5jL-M1St0UrCPcteC9keipvRG0EMDZjFhpLgQb4xWNROdNJhoagaIIgw2b6jIHBvIBLR67bcpoTMrvBKwg:klGXwJS6Dflh2qpJnZEZQw");
        b.AddKeyAttr(143, 2, "2:QJWHvGg-Bcb9Oi0Z:Zv1yCnXKarYtHJXSvH3vmBxQKizEPTyd9USlAI3tUrbM4Wn5mQ4gPfI2c_tzHj_hee6hRtfQqsTa2DVyqDLfkeXezc7D7k2OPOySTiBo0jxc_ufEt7J5heEyrFwcAtjnfWRDoFoqJTS4UDCiwHhejdV2Sj_SYyPbLr8JazYYlgtBEIl7Zc8Ww1JjzCcTf6ReSIEhObJYuSoZwb8U4PzlWK5c24dKn-vEfKE69ymsibecdTi6tXC2iXyoSk9vWiof-bESJcrBcR07JNCkiI2wcubAylHbhVhyTIx5Dcb-4psPXCXrv6CJOXtoChkVBfhzpy-Dqrso5RrQApclWa0ePLxUcWgD1FuSE7YPOQq_bTFgoEu2ljD62SdTl9YONydRHH-niFvPVvxo3r9aprW3-9nKgpB5XoDOMNYsvhDNvjzzZlykvR_B63yB0vFU0cqmCn1KnHBl-6vjD9VamdjeAiuK4ZTRLDSmD7OkndumkDMrdbPrFxVUH8y1FasQYICjHHUHuahS_IeSRspZK_OrBD7zqSZ1hVEtPWUJovCpUzkQpRcxh5sp-x9X90zCXsebtgKa4OhJjBgJyy8PqUGySzgWv76DlcjgS86OoGlwr73nkooYUhR7x3Z8lMaGDJw0P4yzZCimW9p5oe8FvntwtQmyg1Z0v4STMIQAAnJgjAMNztvaBDTIXrjGaGoT6pj_nyUYquHPgZUWANd1L8Bk6AWSdbXF4m51DNFIxgL6iBbpE_JPBzZ8T7wMHOETM4T-0T5rVNfVg_l4OrmGythgN8szJcqwictNzetyiA_ZViGqxJ3EY2MDEh97mhaJkLe2KUMyWxJo8VxMi3DfHbnqYebZ_v7qnfs7mIZYXOiHfdlb5wKs-OKfeP5nttQUhcg22SxoH53tH4jRKNGyXKmWb5eVBdXo3nzvaDLL168_jflQ-anCN651QyP29dTmn_Qup0qJopANu-0TPm1UwdRB-GlW9fJS963x6BWtrfQ-YSIRcoGbUbp5258O4hDZz-lIDIVSoo-iaLbK8aP0qE6IGOhYTvCkBhMkrMfSlsrN9ApJkcw7W3evqw:qjQ7zJWfu51vqOXSDpIORw");
        b.AddKeyAttr(143, 3, "1472218045");
        b.PostProcess();
        TDbCache c = b.Finalize();
        UNIT_ASSERT_EQUAL(1, c.Clients->size());
        b.SetTvmPrivateKey(c, b.Keys(), 1, 1);

        UNIT_ASSERT_EQUAL(143, c.TvmKey->GetId());
    }

    Y_UNIT_TEST(serializePassportKeys) {
        NTvmCommon::TDecryptor decr("BtniSXCXhroPtOgXA61i5ZxSeX/solWR");

        NUnistat::TSignalDiff<> errors("");
        TDbCacheBuilder b(decr, errors);
        b.AddClientAttr(162, 6, "|637|649|661|673|687|699|711|723|735|747|");
        b.AddClientAttr(164, 6, "|639|651|663|675|689|701|713|725|737|749|");
        b.AddClientAttr(224, 6, "|641|653|665|677|691|703|715|727|739|751|");
        b.AddClientAttr(168, 6, "|643|655|667|679|693|705|717|729|741|753|");
        b.AddClientAttr(170, 6, "|645|657|669|681|695|707|719|731|743|755|");
        b.AddClientAttr(188, 6, "|522|");
        b.AddClientAttr(172, 6, "|647|659|671|683|697|709|721|733|745|757|");
        b.AddKeyAttr(637, 1, "2:wp-nTJ_6Ob0JvPSv:ertBhyRKlLwlkpJq3rHlZLhOnoHxfG-7MxAvc2jVJIX_DkQ47iZmE8WHpejG1RbKUkSXFgh7KS3HPzW6qKjOKQtKcYQOromq_DE6FA8rbhjBqKTJkHQme6sagFFHeA3zzH1PPfz3IXYNf88nHv9OBqNQvk9chm2fs_uGNKOysg0Uql2CzrONT4um6oiZN4F-xDUqAcTSJLa8ZwtP52QQGwXJQ_TOv_F8esfunCn8qKE_N2F5GGHxBOdOUMw:6V-PMuvfK2h5iRVDmFP2Dw");
        b.AddKeyAttr(637, 2, "2:8bTo-SpCmkvKfGVi:pjK-5f6NyS6nW7_AeJi2srxcjwZ3I725AGw_Poq8coAlP8icnjxQuwPh7yE_Hsta9Ozk3jK8Q4eCUSMncGf2NWEuA8WbsrAXjjvdMDyw6B_Uq8KgPAzPOG_7CTp5tJLj-7Lpx9r-1-je80iAIYcpLIWIWEN4EcaafDXeID_C1CR4-kCXLGtrt2rDpX0cSvurSbWyGRJm8o8UJ6GluJR5VghxfDAafpn1aQQPKzKV7nTD17ArXUZ_7n6De0Gd-vPDHVJgYg_snbHKZU1FVeyDfvGco6OUXaThgdjvMCeWXXHH7bKHHzXHZ1joHtTR7zwNfMn2mtwvrLwsB65fR0ciERKpGkuFI_k1OkDIDoUfVf2IYDyo0aDBO5_h5xyRkZdLolWMt8BjITetST7cJ0ko2CE_IvBsrTBIugMxtUwMQcPxHXQa3FiDEkKbkIVpUWSwTP5VoR5XV2gcGpyRZZZ0jxrf7njvT8RncqFHswe7MrozmPjopRMJu3ZloINEqKsYtjiwQqS9axxoU3-MQMLMjwPBUM3uL1oo638WzCuIW5N4u5-cMa4jcBQtIqmPYxKZ94I_eMLS-YPZJuELCVuoj1sq9_nVyCKXcInSoQsFFlGHb6_BxIRWHIzw3v3VFs_QMGf9yE6uB5MfHyqWTABegFBZ8jNT97q2HhBlXltqyeeDpYYxCjXGDw_Cg42Y-IoAV4SwZMhZT69YAu8t0sylG0An9wr7scwqkdq64ko-9fzeDEUsD4grdJyZQit6P5qFSGL2G5mxs3ECTE0pKzY_I21KRp-ms6aS16PR3HA8RqojoIyBu_xFLbC_N0WkQ9dPmsmmuH-2ijgq_0j0LyBlGSk2cLwm0Kt-Z__HLxpU9s5-UpKLT4flU7G9-uQ_XGPXV_K-lzachrgwSBzIKap1clIHTUKjvEcfWgTKO8mn4ledARTbRDbNSjljtFBL44ehtQZdJBMWjbspRUutNnWrMEtUrpgYABzmhLmH1T1Whal5uBmRkgrpodrKNl1lnl5Ab7FlpC5OZ-utiChtfCKCBTTEps7SolK9tXA5_JV4wasrJqEjji9Ajyyg0Q:bDAK-hUxPPNLlKhJGq1kTQ");
        b.AddKeyAttr(639, 1, "2:AM5WtVxDcoSFjgRz:PmnzYw_9TPE6vH-6tLY86naOgB6Ern_1WVjNgtCDRUQk7FrmjMlczSveP_lnCYl8tbk_XVSKgTo6wHv-sOWMeDDoNVV_HfH8QZRl_ctKmOjZxV8MzY0vCILiLomkeR6132VpqXor1LX4R0gQ8BQvHVDEm3txvXHJPkkGP-Pk4CpgpN0U3QuALOlaJWIICPYsoc6imO93QKeIGPbkn-ZN1n613tzZKoivrP4aXaojwcddI_yg9iTtDdQOC-Y:UZh7Wq2_8KTAa_kBykmlbg");
        b.AddKeyAttr(639, 2, "2:AosLfWEPv9PUpCmR:NvS7GuP0FcBr1012Q0D-nFTUMifZF_uZb30O7Be6KPQ0KPWaYwsHIcfolquoahye-NbwQbzxJxBLU7Jz1jVAKkjw_GLftjSxaqrd9VMtU1a6djOoLbPYhZgn2vf166F8lk-6aKdiZ1a8JvXu5Xf0pdbv8NeeaRRZgEqXcJO9b8QK4odGN1usCqt-WAbCBKrjiZ0wA3jyAIHYXNXe6FGltNadJh7qg9SQq9hxKvRVRNyEhF-1EIpjnjcDvQoTVhdZD6T_2lsD_8gfXo5-Eha8AbEzPGH-Lp-mkKYOOWVt8ygSdWx0CtgES3H72uQNCcjvTqPJ1FMhShsLEogxJQzxt4Lyibjh4aiIqbvH1Bl4mN26c9PtGlowcy5Rj6HY2d0Fmha9qXK_N5Aq88B0R6Fl2OCKwA_h-0txi9XV-XQXRMqvX5ePPoqIc_rCZ5X2LvVl2vlyG6U95-q0GdJacFrna8xmrx_apcR_NcYG5S1FpTPEQ8MttH2jriQlz1riDAPmjZyKKd8w0QFblliNylZWuwYcVr0DRB6GZtsePAHV6Dzs1sK68ixbPPDEdqY-qWWKXAXoMAE0SNmP_S5FTqsp4sn6ySszCncdz3BCfrJUGcyY8TPaFQS7sO-H0ZcPlv3AiW0vI81OqENdGbtKmntFv25lBpLQgWN6HJzw5LFz4EpI8020WKcPrUiSuh3UFQGeF-EUT5ncwsFRwxdKQdFff8vhvkAdxiLJAoM1yU-ZjF7koT86yoixhWViQz4Al90c26KRx-5GJWaeQnGdbTu1xyn-ucncoACmz-gx0tgBZePC48Mq0ohkGQ57bfjShVWTTaJ7ZYppllg3lJrfKhBf2wYNf5HUnIeIQyuresIWME7syKEAUbtlFdk-ufm0AM6wvOBKuko6E6LJ_45yFW353PF8LepA7Phr40ZFDEoS7lHe3mbC_wcGZjKYVEk20fihSo5FTgknjcyQOoYRjghbgsZtH63w-zOl92TTv1W3j_4D7vR482eG-_ClYaYNVDg_o9-zQrXRpQIOIRUJM5AdcHfIOr0hC3viQ4g_5fL7itkYhjjxwBjzCA:42eVSLUY9C0SaPpesFrcWQ");
        b.AddKeyAttr(641, 1, "2:KfU5Zb6opoxP4Hd5:UiFMmWyP49us_s453wXJo9jHHAno-iv1ltEwf_a9kXS2Zra94-aH1TXwL2029FEIFEfsrpovbEvw1PprwxAnrwH8fT3gU72b2i8psHiEEnYkROaUvsGEphvVAPQMInXW3BvsxcQq5HKkXHZFWghSFR2rmmB64_ecTij_FWM_J_iKn0koCjWwvcxUGexSBR0izO1EwyoniDPFHUwusXeyDWPk9j9hBB64-Wn3NcLPFDAxBNzUnDQS76kspvg:ZBgowP2xI1VMoYl7rtSCWg");
        b.AddKeyAttr(641, 2, "2:MgZZSQcFKfOXTxPV:lSU9JISSoEKJD4KSDebpIYVkPvmUabkOwEco5k_4EjbkRbrSjlsbQ09OjmBKYVdq3Fs6fK7DcFrCm8PqPJqeC0HA5ICDL_JeCpa72khQGdeK_qzLQLMAM4mKooPt1fGEsxapYiz0ud8uCsjY8ywKI-y0EhMVW0PkLGK-NaD2nK7Cc006G305UL_40FlazZGc__mpD_wTBtyiaOxkX0FgRGRWu2aErNa3ksPWjwoqrQz4Bf47_Mv5ralUyQegpOCUbvUD1zh-cp1MYpWwZspchDy6-_tIH1KKVzrbE6TblEhRYxxB89wdt8UjuO1OMLw2wvo1HLrNc1sVDFffcO7lvCpVzCqjceCMOSbVhHqEqn1QWBIvhtZia3r2xwD7mShwe4NTvIJ8yivKNNwxnewg13Uu6dzSIlxsIwH8hF0lNtBlCuWZVxNiAhngWBqLRIB2hy1weCTy6xD4q50Yinwld9FHinXhLafwEkXd1sLSW6uDhpYMd6JHu2sC-ImpTH-B1lJn2HZjccrm0-mFynPm0wd57zY_iufE5NHTo7XfxMW0pVNhkluWo0Ysfy17F8N_uJ5kvgheZlT3E-anEVe9JiI57XRpxZ34cGhulpd3cjDa9cFS9xAq69AFXueN5dq0egeNTbg3qLDYf_OnvhPeCEngN0hRT8kwcsaaEg5glpH1WZpU2Wd8IZ8S928_bjru67AHwxnTmcEebMsldhdS_Nd70qFCG0xJeRU0L0g7Inv62yP54Ka6-M7wBQNDQSEJXV7zqG_NLvcrCR1nYgby47ZAyqhaU3QV8LwO6AYqjm67pkZ14GlVzv9Po9Itii9P1oEmaVf7B2y2dl4GjhESsHlDz5sZWVmt0dEIRKD-uQn6-Jz3MtwDsAXlLVsOhHtT5j0TCrgvricQHtKbYBNqlOtwUMmQTlnyIiJEkksZpAxf1pJjEKVog0bmfGOFl9sWDO--oApAuiuGQQ2wE78n2DVYbP3yGKlP03fy5HeIPyYq5J-VzrBIJIh63NrAWBb_5nIT66ia5RU-cAcJKcqakuyvtYaT8XhhBVJLdx5MlSvA1a7UbmG_5K17aA:dRRutYuH4KfEmItNrgGt1Q");
        b.AddKeyAttr(643, 1, "2:fO374TWz-HNSqS-7:7hQ8vJbocbWhNLa42_I-w_6Vde6JSUZtWou6X2yMqPIj5dSZ1u_GPL5sA5BEyUD7z0Uis3WjyhuLeldHf0MdyoO2HezJa_48etEqmb9yDNIdx0LHormaG53bo9iIbqTQyhH37y-Rti2J0vNJ6Oz0gPPFpnJnfv6bAYjYEW4XEASr4mqKBvy9UVBPiPg3MZ937VxvKg8tbyqw8QH22eZneshqZbkkb3AhP5VOwNQVei4a8flcbE8KEPZR3Q:6jSmcmjtw1RUAILFS79Bgg");
        b.AddKeyAttr(643, 2, "2:Ge2HzfIj5OSitV1-:-P-LPynmBjncBtYDZDEuk7YbvMXo8HS7Zo81EyDuuqHnyMHMbtOp7o2f1YLmCi4Nia_0GtaZ3fbjZSnU5ZTJ9AarKaGs0IlSZTm_aQUnifdwGhE1gaE9jGELx3GLXrpy7LyNqonCmddBY_YI5q8vWsk30-S7OOjDnfO5xD8yD1cBQ7fDS0VK0of3257DH2I689-u4x5buev_V8RhoMa9Xq1MrJvZkGHqWT8FhH3XTl_wXJLNKMhe-Pfx4Tir5f20evlrzfLFM7EbFgGLyHGUn2cFR-Uw2a_Ikz6M-3n8uBeaqB6Kd3jzZYBPoztlK4rI_Nd-dLRPAI7xXpsWtmckUA6flnU-GR_U6SDZ2yqtD2_REAhod3yP9CzIsVqK4ewyQEtUo_X0W3b4hHA08yO9CUcV0yjpFxE1mQs7eN_JDsN-NsYhuYnwbeuUpk5F8ZaE5gNDBSSZQNbbjE-8yjjzeIWNp9WYO1tddeD6rWLrHOlxDr18SyfiAp6ImsNRh5V1ljCDMf3M3Y0nNUAEF4_eBPUt2ppL24fo0C3AMEpbnNe53TRzDbaPihnbeDjnQ1IeMlGBykmSZtV0dZJ4Oe9mEXPc1RNpwCPuTPOgpCYp4GfhHrMSJzVsLXaHF7yCQN8ov8-mDLIkYOhyXWaqxgi-bFDJ0Oqr8asibUSfeD4YnYkyVWmPBzL8f2k8fPd6rSHTs4ZkppBlq-rN1TvHF2NjQ0aN0SNuKXhsCUcbMcuaicLLXufI5wvamdi-I0fH7zIedHsJYiMNujf7Gea1W_S-hGBSxvHo6Ix-Fe0TmyS0RbFRjuEZagfmFY-4Wqut0NyVYQqIAykWwaboVO2s6N8YLu0n5BCi30Q_PSkUPCZK3AXNBPnk5eGe6slSnL-lI2XkoH84ZLrnFuzpEjOrcNOCNGp9SPlvFY1DYZEVI0Yh7ZSwidXXD-oBzIuoBF2jZOyzrXaoITxgB9iScVg8O6DdqJXHylvSlslK6Y4mYlkAAmwoz0JFTqgZNYQGG5BkXLr5j8Fpr9NfI7WGuisZmUz6JYf04myOXhIROWQjEsOS-tOK8LRrjttY:f8xJfWiL__WaM2dbzi-ZcA");
        b.AddKeyAttr(645, 1, "2:Odysu-u5MZp3qJiq:KP5b8ICZH6bn3gYIWjc0k8CvT8UYWGkepo9CQCf04UpH2WBciJiHBIUmxj1kSpW25uD-ZO8PxzU_TNV0qNRfNWcFWI2ZodH3U4EKugQwbRKnaMOQBFpaHOQ07SLT_cwdJFD6hEiSh_EB7aNqE6fGCbDgyoxgg507Oq9tjLPTcA_6rK9suDrZK4E8l01J_gH8lx1FifhqLlW_QGaUb5O0t3ufoE89_0N8x6SA567-n_q7B9gkAgD0ghB5Xj8:wwU3QsuyACPmFgIqYqSexg");
        b.AddKeyAttr(645, 2, "2:DojoZsa9YNtLUiHg:Gd2iKVtUQFbR6x9HRMELsLEvxX_mGznJrRYhu2l3xC_yByvdvALOy7DQLu58YBNdNIkl254VaJ34WoA22EuuLpex_-Mb8phMudds9RSeyxyNYn5n7B0E9hCF_RQ9WHLyhr9jVHFK88eMS2aO6cONLhxydnb8dVoSlDuUv9f_lsrgTFmf2EMueFHlofEkmItNm8aKEI81mDQ87_tAHgVCBaeql_S_hBI-2ZRdMjvoi3FaA1b2-Khgv9633omSxGj3V4g7mpQvghAAdgbKJms9fZQJL6x8x6swR6ASRv17ZxOo0COzi7lhY5GisC0FVqlnVRrh3Totw9xJ3DzT3CG-OEGliKCpNzCcJq2153qJnJIJgGptnMRxcH87eHzB9Ytir5KpEpihHT4ElTlKPk5RLT9QKbNsRZ9wvYyD1p4r9MG_wlSjOnm-wR_d9fIYwjuj0v0FdL7A6ToRgHG0l3cDFfpJgE_Iad_ChDPxosv0Dbcw6cHMqRoHOer73JFGVQjSaXHyrMkLK7StZSxx6-Bu5aQyTaWjkBSPQYg567N_0TqaeG_iSSkitahRcrG3bBdgJxMiX7pmZV7t9LIVWQPkHuFVjyD7n8_o5uBKpy8gRaZ-PcmWws2Gx-4LGIImcxXa-175qMoyYUJzpgi2W2I6YQjvsAXGT7o7BEODt-76kAnGusVFPZ6n01WJTfVrFQYbu-HaexEWJ8cw3Pb705XjhQ9Y4UIB7V_TinV8By3vfoRTsNc2NdW_8jwCn1NR8wBspCvrPT20rnqO8hg3zIaCYXVdp0gygF0BQybgm8yv9VB2HzDrUSrXiVI5VQsjXgGNhWRvFIdbzjsU9vd1iOzWWvIqfCqPSM8u-F_8vEIsc0CkZkAlXAhNJsDO1h4iJp5hyMlD6DBgoJq_Qh4_ccvYKtTbDorV4W9UnqkrpYqkpAX8G0GHx5wnuDE-XYK62ojldHh46jJ44ySZQSVC4vZU_2oQiEgZ8_IkCPly4UzGZSOWsIa4Z0GTKY5HnN7O8-Cty-iNrX-MclMkWKjFCMy6Z1A7TB1Na3oprRO47ttMNtUKsvQLYpn4Aw:aI3BQHG7rW9czCLx9i0rYg");
        b.AddKeyAttr(647, 1, "2:-Q1zFQNz7uSEg2Fd:_8kfRtt_4hE_o2RRbJCC38RnedbORWLQq1Tppgj_DR6XOMhUu4z2R7-wybSHyYA2Ok6q4Ixvd8fZjO--PZnrcwiztCLluC24DpJyfn1_CzrIr90KR0CEUufLU1OZdDWEhUWJuyQZMF2dRrKZGMOoYGT_ROVstu95LyTro9elbWQAn6p0vO43V40zJIazodlv8Zq9tmiOMFLquhUt6k2f5eelR2Bz51qoVftAQQHd1GdpNI6PEyvEFIylusY:GD63I4NAhFkNZaCLgXqJDg");
        b.AddKeyAttr(647, 2, "2:vkl5PBXIIko5Y1Yy:1EBlC-nZ4Tt2oQrbZAGMV47_h6DUZqgbRSOuLSp97XKM9yNTYw8dmihO2rVOBM6z6XHsbLjjQmYqIwMdML8rgyCemQuvXz8HH6dOhvaZOBpoj4rTa1kZtu3maaFmowdDZYZ7h4jBqpJegSgorwxq5l8Iy7-WUh_4bmcxmnF0DsV9kB06dVeef5alzIYanoPrAQDW-yjTOvg9-_q6RjM29Pb5K1CsJOzLaVeDm2F8GWsygT9vlU98Bsf4hk7DwUBNrvhSOfoFEbKdL9euORYmNT3AmHme40VWRleL_MXhSEVScAawjGyqf6YZ3ORtlX_dSKOWQpt7g6PtpM3Q38TEcgEDP89u40BocgZsXMRiydZ72waEz1HVDRfPBx5KKoYa3I4bNxOsianc3FzvARURnyqUHroDxYY8a8CWitdrH_dof21B-n9ZhWZMJCOx6RlKK0F9M3wo1fKS_W3AofuhWgSRPdNIi3DFOBw3q4ofcBJP5dlV80hEKvRaE_dQx2PkA6caM6qyBmqMMYBMhcKj1ordfG1rynz3T8v_PS30Rf1wPBr5lIHtOuTpI9SlhJqHbw3wanzJcxZ1LoRE-7Lct31pY0EyHaspY45UcuAiO_IYde1QbO-r_k_1Ip0feT9-k_bF3CF9JjJ58A-l64SSIpDqihSwkmAVAK0r3tN7pmDLRWoeIEXY_l7s2VhAQuol8UbFXmzI-7mthXO55n-Wmx1nJmDH3LmRlwLMB0RzUsM3qZIKElbHG9ctlcmCqsb0mcUKZTY0Ru-AcUUgHR1QMi_M-JO4syVXKedAzbX6QZEyjs0KJNxOGa_yP5J7ts1tabeOpWY9WO_S92eZHeqdGs1gar28EdZOwRpSGVWg_UnqHfOxgB_RD_cRxwOzv4zQE2JPUq51RW9GLpyHkMu3hjoLMacgG9PFc_TELCAclbrwGRLp2UqATN8y6vTpdXAVipnA7o06WB4qd3za1W9310jVvk6yTzDflFOomZgjjlLp8-37ybsiKA0meexZnM7dmHhE3ic8N7QsQBghxk9x2a8uH3kfolCbj8Rlnx1cyl3Y1IJL08VojWYG:5dKA4WsD0pOFbj-XrKv5ZA");
        b.PostProcess();
        TDbCache c = b.Finalize();
        UNIT_ASSERT_EQUAL(7, c.Clients->size());
        b.SerializePassportKeys(c,
                                b.Keys(),
                                {
                                    .BbProd = 162,
                                    .BbProdYateam = 164,
                                    .BbTest = 224,
                                    .BbTestYateam = 168,
                                    .BbStress = 170,
                                    .BbMimino = 188,
                                    .BbLoad = 190,
                                    .Tvm = 172,
                                },
                                1);

        UNIT_ASSERT_STRINGS_EQUAL("1:CpYBCpEBCP0EEAAahwEwgYQCgYEAg9gn37Gei1h4lTevppfYTMWlWJnoOUumR0-eBCfYMGXiAciyCxZkd30d_jgMg3RSnq9LctElCb1poP7iTTZ8-Z5-TWk2glfNo6TEknAzdCEeXq3xiJ6vL7po5PgbRFV__iQaezM5kr3trj8NodnU-6LJIwGu138eICKj87_pUlUgARAACpYBCpEBCP8EEAAahwEwgYQCgYEAkjKdMRuacq6t3ECeQ3S371wBpwGNfUfaHYcaZEDcdh8tXVEh3uolqd4EtTNqWMxQGpoobFMOneErp5khOUSfyBcWmdSJoO_123wZRO3TH9UqIxu34r0Y6Fg-rDqFb53JEP8WEUNhfDctVta-C9D2nchB9WJjerttNWRvhdxyFKUgARACCpYBCpEBCIEFEAAahwEwgYQCgYEA3HEJnTjW-wxcoIjDXkKS_dvXGe1LM5qhJ7Qb9Ytwh7rBT3B-dNiCSwvGYgmUKQLg5suxdzHK6_jv21t7Oi3DyizDbSAauH38nDCFEESC1D19UxtgFBo0ihCw0KF_9tF_pr1SudIUV-yzZTtK-9Cp3NXz4ltAYSdqV5XzmZt77jUgARABCpUBCpABCIMFEAAahgEwgYMCgYBOlR-0tvg7Ud08RD8gGTpjYxhgCg_qbYGCp6JZKEtpUQ09GA_cnhWHC_MGDtCyKCegfWQLx4WVVPOHuri42A2kTfNki8XZNmP06hzkLZQ0DtgATyvW7HR4bKPX93l8vyBAoXQE0aDOi1VHQQRLoA1ckX9a7dGIt6N0myHkbW-g3SABEAMKlgEKkQEIhQUQABqHATCBhAKBgQDoUiOE2SKFLRajNDshnxs1M0h1yVgw3_4p94TeTzAulJrIJHi46olhceKRXEOVGFrPM_sW9SERlsEpn6NtIIPd0U3op-sokr3Iut-HTQ4iRzUK1NS8TSO4kxuqUHdcWoDtvvDBQf8an5Q_81Mw8J27UibN6qY_R5b5wD4vfCKT1SABEAQSlAEKkQEIhwUQABqHATCBhAKBgQC7uVqYcq7FXR9g8CdGk98SGqYNOWPpPDHGtQ3AYEybwRYrTNOizqnJMtQhRJXvuOOj6Sy84Zb3tpspSh_HQiNtlZj_Gtcs8pmXCFOmwpyCUYKylgJiNU2U0ZwdT9D1Qaa7fiE05kmFjFKgjq-Rypg7iIeUA28-Xk0V4SrxbpJdrSAB",
                                  *c.PublicTvmKeys);
        UNIT_ASSERT_STRINGS_EQUAL("1:CukECuQECP0EEAAa2gQwggJWAoGBAIPYJ9-xnotYeJU3r6aX2EzFpViZ6DlLpkdPngQn2DBl4gHIsgsWZHd9Hf44DIN0Up6vS3LRJQm9aaD-4k02fPmefk1pNoJXzaOkxJJwM3QhHl6t8Yiery-6aOT4G0RVf_4kGnszOZK97a4_DaHZ1PuiySMBrtd_HiAio_O_6VJVAkEAnkxylblCyD8k-Tv5qfnHoK8DzlSFpkP9YGs7efk0smrVQTPSi2JWqiaUN--J9qJJwMeR_YJvvJyRxYqVf7zhKwJBANU331pqHm3p5s4NtJVzqfETSnOlpdDBNuLKlQVzNaZ3XHWAH_fX4nK8PJd5RT4M9rdfRYG3om49LQNjrFrl2n8CQDn3AVD2h5zIOzzCJ4QdMhkYI7iE4x4Ifp1nGe6zZ6l9v-y3iheKiLc8rX2Ig5VvU0P_VTtGsg6sURdReJKMBEoCQBqm--tNQ829PNnBtpKudT4iaU50tLoYJtxZUqCuZrTO646wA_76_E5Xh5LvKKfBntbr6LA29E3HpaBsdYtcu1ACQBPJjlK3KFkH5J8nfzU_OPQV4HnKkLTIf6wNZ28_JpZNWqgmelFsStVE0ob98T7USTgY8j-wTfeTkjixUq_3nCUCQQCWX0iz7ceFx5OUd1ZCptr3kdeY5UezZJMgIfb3fzDTSTxHTEEfkKucHHDqVz1FgdDpNP20SJVUdJw8Muw7skCEAkEA0dAtXLSrbOQr6N0sc7O-Ehc31kSFbA4Ir8vm9VhBfl-yjYkZgsQBX0cb7veTc_2epn27s9sqPSdNRkdRnTnEWyABEAA",
                                  (*c.PrivateBbKeys)[162]);
        UNIT_ASSERT_STRINGS_EQUAL("1:CucECuIECP8EEAAa2AQwggJUAoGBAJIynTEbmnKurdxAnkN0t-9cAacBjX1H2h2HGmRA3HYfLV1RId7qJaneBLUzaljMUBqaKGxTDp3hK6eZITlEn8gXFpnUiaDv9dt8GUTt0x_VKiMbt-K9GOhYPqw6hW-dyRD_FhFDYXw3LVbWvgvQ9p3IQfViY3q7bTVkb4XcchSlAkEAyE0xn0CbQtMEvw9t-Bs8CFIAo8YcbZCIGanN0N9-lJlQufQmD8rWa80_81SjlwtHgMd1DURsMDdvw6Exit-zqwJBALrZ8nTF03OjA1LuA2MfLMjyHbGh15KOHpcoWJIghqSIIvwakAC-7_ypttwEDITmTcQsD5xgufapvltzW-v0CO8CQCVc-9eqnbY14_xwZTeZEBGeOhQClgGeR22GP29LURFyIoHazFHDD7SM54KdLOwzzez8Ggb2XHYsW1O6bPcx_94CQBdbPk6Yum50YGpdwGxj5ZkeQ7Y0OvJRw9LlCxJEENSRBF-DUgAX3f-VNtuAgZCcybiFgfOMFz7VN8tua31-gR4CQBkJpjPoE2haYJfh7b8DZ4EKQBR4w42yEQM1Obob79KTKhc-hMH5Ws15p_5qlHLhaPAY7qGojYYG7fh0JjFb9nUCQDtReDs-sQdqSyspv7r7l3FoWlwhhF216c9wu6ecIVevE7KGlTS9i-rKrWmGCnpmGUND1MnlNwgz1p56iCXab7kCQBNd8xLx5-QAZSI4L2kRPh9n_ZYVNPXiswFo0NLg3K69P5wLTbPqx59t5bGBmQS1usokiPdrP1PgqA4FVY_YqIAgARAC",
                                  (*c.PrivateBbKeys)[164]);
        UNIT_ASSERT_STRINGS_EQUAL("1:CukECuQECIEFEAAa2gQwggJWAoGBANxxCZ041vsMXKCIw15Ckv3b1xntSzOaoSe0G_WLcIe6wU9wfnTYgksLxmIJlCkC4ObLsXcxyuv479tbezotw8osw20gGrh9_JwwhRBEgtQ9fVMbYBQaNIoQsNChf_bRf6a9UrnSFFfss2U7SvvQqdzV8-JbQGEnaleV85mbe-41AkEA30j2RUkD8-HfqK8kuDULo2OhD8wMFAuQSpVW5gFMajrOav--JWVvLE3Dx04qwcmLEm7x1-I4PtO133JOaa378wJBAPy9bCUr_qWnFa7AeiZ31omkr8owi3TrdKz-jhEAH0iW0pnw2TzXJZH2Hr8YbT4TuZN3V49-kGOxjJU6Yjn-PzcCQASGPOKHyJLdT8eBvf37oMIjVXWUYlsFcYpIGT6FEsGT0jar4O6R2AKGdz6Nh2sre3teBzksZE69TmITEIb7YT4CQB-XrYSlf9S04rXYD0TO-tE0lflGEW6dbpWf0cIgA-kS2lM-Gyea5LI-w9fjDafCdzJu6vHv0gx2MZKnTEc_x-cCQBvpHsipIH58O_UV5JcGoXRsdCH5gYKBcglSqtzAKY1HWc1f98SsreWJuHjpxVg5MWJN3jr8RwfadrvuSc01v34CQQCPNlcQVDnKsWQ72MBXT6ojaLhybvmgVfQod-ZN98qz0iybXuL_1BCI-SxH6a1J1mCK2j-bo6h4fHvFqdRvW7f4AkEAkZCCLTXJSJrqQOBTJO9UyHu27dX8RolMEIC82JT-J15E1GnDHdrijIXxrwAiaCvp6CGgIULtH5olgzc7dVknwyABEAE",
                                  (*c.PrivateBbKeys)[224]);
        UNIT_ASSERT_STRINGS_EQUAL("1:CuYECuEECIMFEAAa1wQwggJTAoGATpUftLb4O1HdPEQ_IBk6Y2MYYAoP6m2BgqeiWShLaVENPRgP3J4VhwvzBg7QsignoH1kC8eFlVTzh7q4uNgNpE3zZIvF2TZj9Ooc5C2UNA7YAE8r1ux0eGyj1_d5fL8gQKF0BNGgzotVR0EES6ANXJF_Wu3RiLejdJsh5G1voN0CQQCCc_ayQrGeUsmAQrkqM7Hu6ab5YZX_80hdaIQ8viotkeX6kjN_KA6-WBGlfwXEziBBSE3W1_KN9IdisLdMvSbTAkEAmjWzJ-E_oy6G2ntCA6C6h2Zjqg4qULYASeE28oRnuFxRsTe4G_21PCj2OZKLmIuw4idTG-asUjE9MEe_VsirjwJATDX-uZ83BkWzGgrOIXpgBeSaFfDX_mDfBfTWYV2QBhsOvOsx3_3zcI902AVCx0CyKRNDN9yqNDTIn9DeaXX_RwJAE0a2ZPwn9GXQ209oQHQXUOzMdUHFShbACTwm3lCM9wuKNib3A3-2p4UexzJRcxF2HETqY3zVikYnpgj36tkVcgJAEE5-1khWM8pZMAhXJUZ2Pd003ywyv_5pC60Qh5fFRbI8v1JGb-UB18sCNK_guJnECCkJutr-Ub6Q7FYW6Zek2gJAIwJW9jaNegg_KuuFPYsRSfDOsIAVYpOPb30Nn_qZBuuknNmQksQ2kSjUGxeQWG8d8lGA-bnSvxu-u3fHi2GTKgJAZJ0rJnPrDjmqoMtBnCXSRcVEUlFd8hgtUwZHvbh6kDpXcVSfvOqt_n4VXHy5SNgAzYS44EyiFpZUFNn1t0EJxyABEAM",
                                  (*c.PrivateBbKeys)[168]);
        UNIT_ASSERT_STRINGS_EQUAL("1:CucECuIECIUFEAAa2AQwggJUAoGBAOhSI4TZIoUtFqM0OyGfGzUzSHXJWDDf_in3hN5PMC6UmsgkeLjqiWFx4pFcQ5UYWs8z-xb1IRGWwSmfo20gg93RTein6yiSvci634dNDiJHNQrU1LxNI7iTG6pQd1xagO2-8MFB_xqflD_zUzDwnbtSJs3qpj9HlvnAPi98IpPVAkEA74tuVqXbMKxrMiTV-txPu91QZOuo8ZZ-LXVD5vz6FEWZQnxEmSGTjKFVaPLfF9hxN5PHmbJXPILwZmi5ngic6wJBAPhHrAPgEL7JEwRP81JZ7o8JMC952bw3KCIrnNVT6nhcYTh-xuHsYv--SBZZ99Zhzj3LE34fVASCmBuyj39WYj8CQCC4tJjkwdgAyz0IfTqjdh44IQ8dWVb4fY0e0Vhk3kYERiHgsRtuL-hzqGLfLBoMc4zxR_3BfGL0xnhqIkyJYikCQB8I9YB8AhfZImCJ_mpLPdHhJgXvOzeG5QRFc5qqfU8LjCcP2Nw9jF_3yQLLPvrMOce5Ym_D6oCQUwN2Ue_qzEgCQB3xbcrUu2YVjWZEmr9bifd7qgyddR4yz8WuqHzfn0KIsyhPiJMkMnGUKq0eW-L7DibyePM2SueQXgzNFzPBE50CQBUEBZBdYg7thD3E5F6y_T7ACyA2Pgfcu8cKjEifTGusyVDp-my7MuJ2tjeItrPmIEAPgm1GctEREct0J-VtVMkCQC-c28ip4N-65QVL88uKXT4Wf_fWMovq5eweRys3lLnmJMPwhcJTCkO9cf7825ZgFf-dm04BsELpidfO0BlBpWsgARAE",
                                  (*c.PrivateBbKeys)[170]);
        UNIT_ASSERT_STRINGS_EQUAL("1:CukECuQECP0EEAAa2gQwggJWAoGBAIPYJ9-xnotYeJU3r6aX2EzFpViZ6DlLpkdPngQn2DBl4gHIsgsWZHd9Hf44DIN0Up6vS3LRJQm9aaD-4k02fPmefk1pNoJXzaOkxJJwM3QhHl6t8Yiery-6aOT4G0RVf_4kGnszOZK97a4_DaHZ1PuiySMBrtd_HiAio_O_6VJVAkEAnkxylblCyD8k-Tv5qfnHoK8DzlSFpkP9YGs7efk0smrVQTPSi2JWqiaUN--J9qJJwMeR_YJvvJyRxYqVf7zhKwJBANU331pqHm3p5s4NtJVzqfETSnOlpdDBNuLKlQVzNaZ3XHWAH_fX4nK8PJd5RT4M9rdfRYG3om49LQNjrFrl2n8CQDn3AVD2h5zIOzzCJ4QdMhkYI7iE4x4Ifp1nGe6zZ6l9v-y3iheKiLc8rX2Ig5VvU0P_VTtGsg6sURdReJKMBEoCQBqm--tNQ829PNnBtpKudT4iaU50tLoYJtxZUqCuZrTO646wA_76_E5Xh5LvKKfBntbr6LA29E3HpaBsdYtcu1ACQBPJjlK3KFkH5J8nfzU_OPQV4HnKkLTIf6wNZ28_JpZNWqgmelFsStVE0ob98T7USTgY8j-wTfeTkjixUq_3nCUCQQCWX0iz7ceFx5OUd1ZCptr3kdeY5UezZJMgIfb3fzDTSTxHTEEfkKucHHDqVz1FgdDpNP20SJVUdJw8Muw7skCEAkEA0dAtXLSrbOQr6N0sc7O-Ehc31kSFbA4Ir8vm9VhBfl-yjYkZgsQBX0cb7veTc_2epn27s9sqPSdNRkdRnTnEWyABEAA",
                                  (*c.PrivateBbKeys)[188]);
        UNIT_ASSERT_STRINGS_EQUAL("1:CucECuIECIUFEAAa2AQwggJUAoGBAOhSI4TZIoUtFqM0OyGfGzUzSHXJWDDf_in3hN5PMC6UmsgkeLjqiWFx4pFcQ5UYWs8z-xb1IRGWwSmfo20gg93RTein6yiSvci634dNDiJHNQrU1LxNI7iTG6pQd1xagO2-8MFB_xqflD_zUzDwnbtSJs3qpj9HlvnAPi98IpPVAkEA74tuVqXbMKxrMiTV-txPu91QZOuo8ZZ-LXVD5vz6FEWZQnxEmSGTjKFVaPLfF9hxN5PHmbJXPILwZmi5ngic6wJBAPhHrAPgEL7JEwRP81JZ7o8JMC952bw3KCIrnNVT6nhcYTh-xuHsYv--SBZZ99Zhzj3LE34fVASCmBuyj39WYj8CQCC4tJjkwdgAyz0IfTqjdh44IQ8dWVb4fY0e0Vhk3kYERiHgsRtuL-hzqGLfLBoMc4zxR_3BfGL0xnhqIkyJYikCQB8I9YB8AhfZImCJ_mpLPdHhJgXvOzeG5QRFc5qqfU8LjCcP2Nw9jF_3yQLLPvrMOce5Ym_D6oCQUwN2Ue_qzEgCQB3xbcrUu2YVjWZEmr9bifd7qgyddR4yz8WuqHzfn0KIsyhPiJMkMnGUKq0eW-L7DibyePM2SueQXgzNFzPBE50CQBUEBZBdYg7thD3E5F6y_T7ACyA2Pgfcu8cKjEifTGusyVDp-my7MuJ2tjeItrPmIEAPgm1GctEREct0J-VtVMkCQC-c28ip4N-65QVL88uKXT4Wf_fWMovq5eweRys3lLnmJMPwhcJTCkO9cf7825ZgFf-dm04BsELpidfO0BlBpWsgARAE",
                                  (*c.PrivateBbKeys)[190]);
        UNIT_ASSERT_STRINGS_EQUAL("",
                                  (*c.PrivateBbKeys)[172]);
    }

    Y_UNIT_TEST(serialize) {
        NTvmCommon::TDecryptor decr("BtniSXCXhroPtOgXA61i5ZxSeX/solWR");

        NUnistat::TSignalDiff<> errors("");
        TDbCacheBuilder b(decr, errors, 1);
        b.AddClientAttr(162, 6, "|637|649|661|673|687|699|711|723|735|747|");
        b.AddClientAttr(164, 6, "|639|651|663|675|689|701|713|725|737|749|");
        b.AddClientAttr(224, 6, "|641|653|665|677|691|703|715|727|739|751|");
        b.AddClientAttr(168, 6, "|643|655|667|679|693|705|717|729|741|753|");
        b.AddClientAttr(170, 6, "|645|657|669|681|695|707|719|731|743|755|");
        b.AddClientAttr(188, 6, "|522|");
        b.AddClientAttr(172, 6, "|647|659|671|683|697|709|721|733|745|757|");
        b.AddKeyAttr(637, 1, "2:wp-nTJ_6Ob0JvPSv:ertBhyRKlLwlkpJq3rHlZLhOnoHxfG-7MxAvc2jVJIX_DkQ47iZmE8WHpejG1RbKUkSXFgh7KS3HPzW6qKjOKQtKcYQOromq_DE6FA8rbhjBqKTJkHQme6sagFFHeA3zzH1PPfz3IXYNf88nHv9OBqNQvk9chm2fs_uGNKOysg0Uql2CzrONT4um6oiZN4F-xDUqAcTSJLa8ZwtP52QQGwXJQ_TOv_F8esfunCn8qKE_N2F5GGHxBOdOUMw:6V-PMuvfK2h5iRVDmFP2Dw");
        b.AddKeyAttr(637, 2, "2:8bTo-SpCmkvKfGVi:pjK-5f6NyS6nW7_AeJi2srxcjwZ3I725AGw_Poq8coAlP8icnjxQuwPh7yE_Hsta9Ozk3jK8Q4eCUSMncGf2NWEuA8WbsrAXjjvdMDyw6B_Uq8KgPAzPOG_7CTp5tJLj-7Lpx9r-1-je80iAIYcpLIWIWEN4EcaafDXeID_C1CR4-kCXLGtrt2rDpX0cSvurSbWyGRJm8o8UJ6GluJR5VghxfDAafpn1aQQPKzKV7nTD17ArXUZ_7n6De0Gd-vPDHVJgYg_snbHKZU1FVeyDfvGco6OUXaThgdjvMCeWXXHH7bKHHzXHZ1joHtTR7zwNfMn2mtwvrLwsB65fR0ciERKpGkuFI_k1OkDIDoUfVf2IYDyo0aDBO5_h5xyRkZdLolWMt8BjITetST7cJ0ko2CE_IvBsrTBIugMxtUwMQcPxHXQa3FiDEkKbkIVpUWSwTP5VoR5XV2gcGpyRZZZ0jxrf7njvT8RncqFHswe7MrozmPjopRMJu3ZloINEqKsYtjiwQqS9axxoU3-MQMLMjwPBUM3uL1oo638WzCuIW5N4u5-cMa4jcBQtIqmPYxKZ94I_eMLS-YPZJuELCVuoj1sq9_nVyCKXcInSoQsFFlGHb6_BxIRWHIzw3v3VFs_QMGf9yE6uB5MfHyqWTABegFBZ8jNT97q2HhBlXltqyeeDpYYxCjXGDw_Cg42Y-IoAV4SwZMhZT69YAu8t0sylG0An9wr7scwqkdq64ko-9fzeDEUsD4grdJyZQit6P5qFSGL2G5mxs3ECTE0pKzY_I21KRp-ms6aS16PR3HA8RqojoIyBu_xFLbC_N0WkQ9dPmsmmuH-2ijgq_0j0LyBlGSk2cLwm0Kt-Z__HLxpU9s5-UpKLT4flU7G9-uQ_XGPXV_K-lzachrgwSBzIKap1clIHTUKjvEcfWgTKO8mn4ledARTbRDbNSjljtFBL44ehtQZdJBMWjbspRUutNnWrMEtUrpgYABzmhLmH1T1Whal5uBmRkgrpodrKNl1lnl5Ab7FlpC5OZ-utiChtfCKCBTTEps7SolK9tXA5_JV4wasrJqEjji9Ajyyg0Q:bDAK-hUxPPNLlKhJGq1kTQ");
        b.AddKeyAttr(639, 1, "2:AM5WtVxDcoSFjgRz:PmnzYw_9TPE6vH-6tLY86naOgB6Ern_1WVjNgtCDRUQk7FrmjMlczSveP_lnCYl8tbk_XVSKgTo6wHv-sOWMeDDoNVV_HfH8QZRl_ctKmOjZxV8MzY0vCILiLomkeR6132VpqXor1LX4R0gQ8BQvHVDEm3txvXHJPkkGP-Pk4CpgpN0U3QuALOlaJWIICPYsoc6imO93QKeIGPbkn-ZN1n613tzZKoivrP4aXaojwcddI_yg9iTtDdQOC-Y:UZh7Wq2_8KTAa_kBykmlbg");
        b.AddKeyAttr(639, 2, "2:AosLfWEPv9PUpCmR:NvS7GuP0FcBr1012Q0D-nFTUMifZF_uZb30O7Be6KPQ0KPWaYwsHIcfolquoahye-NbwQbzxJxBLU7Jz1jVAKkjw_GLftjSxaqrd9VMtU1a6djOoLbPYhZgn2vf166F8lk-6aKdiZ1a8JvXu5Xf0pdbv8NeeaRRZgEqXcJO9b8QK4odGN1usCqt-WAbCBKrjiZ0wA3jyAIHYXNXe6FGltNadJh7qg9SQq9hxKvRVRNyEhF-1EIpjnjcDvQoTVhdZD6T_2lsD_8gfXo5-Eha8AbEzPGH-Lp-mkKYOOWVt8ygSdWx0CtgES3H72uQNCcjvTqPJ1FMhShsLEogxJQzxt4Lyibjh4aiIqbvH1Bl4mN26c9PtGlowcy5Rj6HY2d0Fmha9qXK_N5Aq88B0R6Fl2OCKwA_h-0txi9XV-XQXRMqvX5ePPoqIc_rCZ5X2LvVl2vlyG6U95-q0GdJacFrna8xmrx_apcR_NcYG5S1FpTPEQ8MttH2jriQlz1riDAPmjZyKKd8w0QFblliNylZWuwYcVr0DRB6GZtsePAHV6Dzs1sK68ixbPPDEdqY-qWWKXAXoMAE0SNmP_S5FTqsp4sn6ySszCncdz3BCfrJUGcyY8TPaFQS7sO-H0ZcPlv3AiW0vI81OqENdGbtKmntFv25lBpLQgWN6HJzw5LFz4EpI8020WKcPrUiSuh3UFQGeF-EUT5ncwsFRwxdKQdFff8vhvkAdxiLJAoM1yU-ZjF7koT86yoixhWViQz4Al90c26KRx-5GJWaeQnGdbTu1xyn-ucncoACmz-gx0tgBZePC48Mq0ohkGQ57bfjShVWTTaJ7ZYppllg3lJrfKhBf2wYNf5HUnIeIQyuresIWME7syKEAUbtlFdk-ufm0AM6wvOBKuko6E6LJ_45yFW353PF8LepA7Phr40ZFDEoS7lHe3mbC_wcGZjKYVEk20fihSo5FTgknjcyQOoYRjghbgsZtH63w-zOl92TTv1W3j_4D7vR482eG-_ClYaYNVDg_o9-zQrXRpQIOIRUJM5AdcHfIOr0hC3viQ4g_5fL7itkYhjjxwBjzCA:42eVSLUY9C0SaPpesFrcWQ");
        b.AddKeyAttr(641, 1, "2:KfU5Zb6opoxP4Hd5:UiFMmWyP49us_s453wXJo9jHHAno-iv1ltEwf_a9kXS2Zra94-aH1TXwL2029FEIFEfsrpovbEvw1PprwxAnrwH8fT3gU72b2i8psHiEEnYkROaUvsGEphvVAPQMInXW3BvsxcQq5HKkXHZFWghSFR2rmmB64_ecTij_FWM_J_iKn0koCjWwvcxUGexSBR0izO1EwyoniDPFHUwusXeyDWPk9j9hBB64-Wn3NcLPFDAxBNzUnDQS76kspvg:ZBgowP2xI1VMoYl7rtSCWg");
        b.AddKeyAttr(641, 2, "2:MgZZSQcFKfOXTxPV:lSU9JISSoEKJD4KSDebpIYVkPvmUabkOwEco5k_4EjbkRbrSjlsbQ09OjmBKYVdq3Fs6fK7DcFrCm8PqPJqeC0HA5ICDL_JeCpa72khQGdeK_qzLQLMAM4mKooPt1fGEsxapYiz0ud8uCsjY8ywKI-y0EhMVW0PkLGK-NaD2nK7Cc006G305UL_40FlazZGc__mpD_wTBtyiaOxkX0FgRGRWu2aErNa3ksPWjwoqrQz4Bf47_Mv5ralUyQegpOCUbvUD1zh-cp1MYpWwZspchDy6-_tIH1KKVzrbE6TblEhRYxxB89wdt8UjuO1OMLw2wvo1HLrNc1sVDFffcO7lvCpVzCqjceCMOSbVhHqEqn1QWBIvhtZia3r2xwD7mShwe4NTvIJ8yivKNNwxnewg13Uu6dzSIlxsIwH8hF0lNtBlCuWZVxNiAhngWBqLRIB2hy1weCTy6xD4q50Yinwld9FHinXhLafwEkXd1sLSW6uDhpYMd6JHu2sC-ImpTH-B1lJn2HZjccrm0-mFynPm0wd57zY_iufE5NHTo7XfxMW0pVNhkluWo0Ysfy17F8N_uJ5kvgheZlT3E-anEVe9JiI57XRpxZ34cGhulpd3cjDa9cFS9xAq69AFXueN5dq0egeNTbg3qLDYf_OnvhPeCEngN0hRT8kwcsaaEg5glpH1WZpU2Wd8IZ8S928_bjru67AHwxnTmcEebMsldhdS_Nd70qFCG0xJeRU0L0g7Inv62yP54Ka6-M7wBQNDQSEJXV7zqG_NLvcrCR1nYgby47ZAyqhaU3QV8LwO6AYqjm67pkZ14GlVzv9Po9Itii9P1oEmaVf7B2y2dl4GjhESsHlDz5sZWVmt0dEIRKD-uQn6-Jz3MtwDsAXlLVsOhHtT5j0TCrgvricQHtKbYBNqlOtwUMmQTlnyIiJEkksZpAxf1pJjEKVog0bmfGOFl9sWDO--oApAuiuGQQ2wE78n2DVYbP3yGKlP03fy5HeIPyYq5J-VzrBIJIh63NrAWBb_5nIT66ia5RU-cAcJKcqakuyvtYaT8XhhBVJLdx5MlSvA1a7UbmG_5K17aA:dRRutYuH4KfEmItNrgGt1Q");
        b.AddKeyAttr(643, 1, "2:fO374TWz-HNSqS-7:7hQ8vJbocbWhNLa42_I-w_6Vde6JSUZtWou6X2yMqPIj5dSZ1u_GPL5sA5BEyUD7z0Uis3WjyhuLeldHf0MdyoO2HezJa_48etEqmb9yDNIdx0LHormaG53bo9iIbqTQyhH37y-Rti2J0vNJ6Oz0gPPFpnJnfv6bAYjYEW4XEASr4mqKBvy9UVBPiPg3MZ937VxvKg8tbyqw8QH22eZneshqZbkkb3AhP5VOwNQVei4a8flcbE8KEPZR3Q:6jSmcmjtw1RUAILFS79Bgg");
        b.AddKeyAttr(643, 2, "2:Ge2HzfIj5OSitV1-:-P-LPynmBjncBtYDZDEuk7YbvMXo8HS7Zo81EyDuuqHnyMHMbtOp7o2f1YLmCi4Nia_0GtaZ3fbjZSnU5ZTJ9AarKaGs0IlSZTm_aQUnifdwGhE1gaE9jGELx3GLXrpy7LyNqonCmddBY_YI5q8vWsk30-S7OOjDnfO5xD8yD1cBQ7fDS0VK0of3257DH2I689-u4x5buev_V8RhoMa9Xq1MrJvZkGHqWT8FhH3XTl_wXJLNKMhe-Pfx4Tir5f20evlrzfLFM7EbFgGLyHGUn2cFR-Uw2a_Ikz6M-3n8uBeaqB6Kd3jzZYBPoztlK4rI_Nd-dLRPAI7xXpsWtmckUA6flnU-GR_U6SDZ2yqtD2_REAhod3yP9CzIsVqK4ewyQEtUo_X0W3b4hHA08yO9CUcV0yjpFxE1mQs7eN_JDsN-NsYhuYnwbeuUpk5F8ZaE5gNDBSSZQNbbjE-8yjjzeIWNp9WYO1tddeD6rWLrHOlxDr18SyfiAp6ImsNRh5V1ljCDMf3M3Y0nNUAEF4_eBPUt2ppL24fo0C3AMEpbnNe53TRzDbaPihnbeDjnQ1IeMlGBykmSZtV0dZJ4Oe9mEXPc1RNpwCPuTPOgpCYp4GfhHrMSJzVsLXaHF7yCQN8ov8-mDLIkYOhyXWaqxgi-bFDJ0Oqr8asibUSfeD4YnYkyVWmPBzL8f2k8fPd6rSHTs4ZkppBlq-rN1TvHF2NjQ0aN0SNuKXhsCUcbMcuaicLLXufI5wvamdi-I0fH7zIedHsJYiMNujf7Gea1W_S-hGBSxvHo6Ix-Fe0TmyS0RbFRjuEZagfmFY-4Wqut0NyVYQqIAykWwaboVO2s6N8YLu0n5BCi30Q_PSkUPCZK3AXNBPnk5eGe6slSnL-lI2XkoH84ZLrnFuzpEjOrcNOCNGp9SPlvFY1DYZEVI0Yh7ZSwidXXD-oBzIuoBF2jZOyzrXaoITxgB9iScVg8O6DdqJXHylvSlslK6Y4mYlkAAmwoz0JFTqgZNYQGG5BkXLr5j8Fpr9NfI7WGuisZmUz6JYf04myOXhIROWQjEsOS-tOK8LRrjttY:f8xJfWiL__WaM2dbzi-ZcA");
        b.AddKeyAttr(645, 1, "2:Odysu-u5MZp3qJiq:KP5b8ICZH6bn3gYIWjc0k8CvT8UYWGkepo9CQCf04UpH2WBciJiHBIUmxj1kSpW25uD-ZO8PxzU_TNV0qNRfNWcFWI2ZodH3U4EKugQwbRKnaMOQBFpaHOQ07SLT_cwdJFD6hEiSh_EB7aNqE6fGCbDgyoxgg507Oq9tjLPTcA_6rK9suDrZK4E8l01J_gH8lx1FifhqLlW_QGaUb5O0t3ufoE89_0N8x6SA567-n_q7B9gkAgD0ghB5Xj8:wwU3QsuyACPmFgIqYqSexg");
        b.AddKeyAttr(645, 2, "2:DojoZsa9YNtLUiHg:Gd2iKVtUQFbR6x9HRMELsLEvxX_mGznJrRYhu2l3xC_yByvdvALOy7DQLu58YBNdNIkl254VaJ34WoA22EuuLpex_-Mb8phMudds9RSeyxyNYn5n7B0E9hCF_RQ9WHLyhr9jVHFK88eMS2aO6cONLhxydnb8dVoSlDuUv9f_lsrgTFmf2EMueFHlofEkmItNm8aKEI81mDQ87_tAHgVCBaeql_S_hBI-2ZRdMjvoi3FaA1b2-Khgv9633omSxGj3V4g7mpQvghAAdgbKJms9fZQJL6x8x6swR6ASRv17ZxOo0COzi7lhY5GisC0FVqlnVRrh3Totw9xJ3DzT3CG-OEGliKCpNzCcJq2153qJnJIJgGptnMRxcH87eHzB9Ytir5KpEpihHT4ElTlKPk5RLT9QKbNsRZ9wvYyD1p4r9MG_wlSjOnm-wR_d9fIYwjuj0v0FdL7A6ToRgHG0l3cDFfpJgE_Iad_ChDPxosv0Dbcw6cHMqRoHOer73JFGVQjSaXHyrMkLK7StZSxx6-Bu5aQyTaWjkBSPQYg567N_0TqaeG_iSSkitahRcrG3bBdgJxMiX7pmZV7t9LIVWQPkHuFVjyD7n8_o5uBKpy8gRaZ-PcmWws2Gx-4LGIImcxXa-175qMoyYUJzpgi2W2I6YQjvsAXGT7o7BEODt-76kAnGusVFPZ6n01WJTfVrFQYbu-HaexEWJ8cw3Pb705XjhQ9Y4UIB7V_TinV8By3vfoRTsNc2NdW_8jwCn1NR8wBspCvrPT20rnqO8hg3zIaCYXVdp0gygF0BQybgm8yv9VB2HzDrUSrXiVI5VQsjXgGNhWRvFIdbzjsU9vd1iOzWWvIqfCqPSM8u-F_8vEIsc0CkZkAlXAhNJsDO1h4iJp5hyMlD6DBgoJq_Qh4_ccvYKtTbDorV4W9UnqkrpYqkpAX8G0GHx5wnuDE-XYK62ojldHh46jJ44ySZQSVC4vZU_2oQiEgZ8_IkCPly4UzGZSOWsIa4Z0GTKY5HnN7O8-Cty-iNrX-MclMkWKjFCMy6Z1A7TB1Na3oprRO47ttMNtUKsvQLYpn4Aw:aI3BQHG7rW9czCLx9i0rYg");
        b.AddKeyAttr(647, 1, "2:-Q1zFQNz7uSEg2Fd:_8kfRtt_4hE_o2RRbJCC38RnedbORWLQq1Tppgj_DR6XOMhUu4z2R7-wybSHyYA2Ok6q4Ixvd8fZjO--PZnrcwiztCLluC24DpJyfn1_CzrIr90KR0CEUufLU1OZdDWEhUWJuyQZMF2dRrKZGMOoYGT_ROVstu95LyTro9elbWQAn6p0vO43V40zJIazodlv8Zq9tmiOMFLquhUt6k2f5eelR2Bz51qoVftAQQHd1GdpNI6PEyvEFIylusY:GD63I4NAhFkNZaCLgXqJDg");
        b.AddKeyAttr(647, 2, "2:vkl5PBXIIko5Y1Yy:1EBlC-nZ4Tt2oQrbZAGMV47_h6DUZqgbRSOuLSp97XKM9yNTYw8dmihO2rVOBM6z6XHsbLjjQmYqIwMdML8rgyCemQuvXz8HH6dOhvaZOBpoj4rTa1kZtu3maaFmowdDZYZ7h4jBqpJegSgorwxq5l8Iy7-WUh_4bmcxmnF0DsV9kB06dVeef5alzIYanoPrAQDW-yjTOvg9-_q6RjM29Pb5K1CsJOzLaVeDm2F8GWsygT9vlU98Bsf4hk7DwUBNrvhSOfoFEbKdL9euORYmNT3AmHme40VWRleL_MXhSEVScAawjGyqf6YZ3ORtlX_dSKOWQpt7g6PtpM3Q38TEcgEDP89u40BocgZsXMRiydZ72waEz1HVDRfPBx5KKoYa3I4bNxOsianc3FzvARURnyqUHroDxYY8a8CWitdrH_dof21B-n9ZhWZMJCOx6RlKK0F9M3wo1fKS_W3AofuhWgSRPdNIi3DFOBw3q4ofcBJP5dlV80hEKvRaE_dQx2PkA6caM6qyBmqMMYBMhcKj1ordfG1rynz3T8v_PS30Rf1wPBr5lIHtOuTpI9SlhJqHbw3wanzJcxZ1LoRE-7Lct31pY0EyHaspY45UcuAiO_IYde1QbO-r_k_1Ip0feT9-k_bF3CF9JjJ58A-l64SSIpDqihSwkmAVAK0r3tN7pmDLRWoeIEXY_l7s2VhAQuol8UbFXmzI-7mthXO55n-Wmx1nJmDH3LmRlwLMB0RzUsM3qZIKElbHG9ctlcmCqsb0mcUKZTY0Ru-AcUUgHR1QMi_M-JO4syVXKedAzbX6QZEyjs0KJNxOGa_yP5J7ts1tabeOpWY9WO_S92eZHeqdGs1gar28EdZOwRpSGVWg_UnqHfOxgB_RD_cRxwOzv4zQE2JPUq51RW9GLpyHkMu3hjoLMacgG9PFc_TELCAclbrwGRLp2UqATN8y6vTpdXAVipnA7o06WB4qd3za1W9310jVvk6yTzDflFOomZgjjlLp8-37ybsiKA0meexZnM7dmHhE3ic8N7QsQBghxk9x2a8uH3kfolCbj8Rlnx1cyl3Y1IJL08VojWYG:5dKA4WsD0pOFbj-XrKv5ZA");
        b.PostProcess();
        b.Serialize(1, {
                           .BbProd = 162,
                           .BbProdYateam = 164,
                           .BbTest = 224,
                           .BbTestYateam = 168,
                           .BbStress = 170,
                           .BbMimino = 188,
                           .BbLoad = 190,
                           .Tvm = 172,
                       });
        TDbCache c = b.Finalize();

        UNIT_ASSERT_STRINGS_EQUAL("1:CpYBCpEBCP0EEAAahwEwgYQCgYEAg9gn37Gei1h4lTevppfYTMWlWJnoOUumR0-eBCfYMGXiAciyCxZkd30d_jgMg3RSnq9LctElCb1poP7iTTZ8-Z5-TWk2glfNo6TEknAzdCEeXq3xiJ6vL7po5PgbRFV__iQaezM5kr3trj8NodnU-6LJIwGu138eICKj87_pUlUgARAACpYBCpEBCP8EEAAahwEwgYQCgYEAkjKdMRuacq6t3ECeQ3S371wBpwGNfUfaHYcaZEDcdh8tXVEh3uolqd4EtTNqWMxQGpoobFMOneErp5khOUSfyBcWmdSJoO_123wZRO3TH9UqIxu34r0Y6Fg-rDqFb53JEP8WEUNhfDctVta-C9D2nchB9WJjerttNWRvhdxyFKUgARACCpYBCpEBCIEFEAAahwEwgYQCgYEA3HEJnTjW-wxcoIjDXkKS_dvXGe1LM5qhJ7Qb9Ytwh7rBT3B-dNiCSwvGYgmUKQLg5suxdzHK6_jv21t7Oi3DyizDbSAauH38nDCFEESC1D19UxtgFBo0ihCw0KF_9tF_pr1SudIUV-yzZTtK-9Cp3NXz4ltAYSdqV5XzmZt77jUgARABCpUBCpABCIMFEAAahgEwgYMCgYBOlR-0tvg7Ud08RD8gGTpjYxhgCg_qbYGCp6JZKEtpUQ09GA_cnhWHC_MGDtCyKCegfWQLx4WVVPOHuri42A2kTfNki8XZNmP06hzkLZQ0DtgATyvW7HR4bKPX93l8vyBAoXQE0aDOi1VHQQRLoA1ckX9a7dGIt6N0myHkbW-g3SABEAMKlgEKkQEIhQUQABqHATCBhAKBgQDoUiOE2SKFLRajNDshnxs1M0h1yVgw3_4p94TeTzAulJrIJHi46olhceKRXEOVGFrPM_sW9SERlsEpn6NtIIPd0U3op-sokr3Iut-HTQ4iRzUK1NS8TSO4kxuqUHdcWoDtvvDBQf8an5Q_81Mw8J27UibN6qY_R5b5wD4vfCKT1SABEAQSlAEKkQEIhwUQABqHATCBhAKBgQC7uVqYcq7FXR9g8CdGk98SGqYNOWPpPDHGtQ3AYEybwRYrTNOizqnJMtQhRJXvuOOj6Sy84Zb3tpspSh_HQiNtlZj_Gtcs8pmXCFOmwpyCUYKylgJiNU2U0ZwdT9D1Qaa7fiE05kmFjFKgjq-Rypg7iIeUA28-Xk0V4SrxbpJdrSAB",
                                  *c.PublicTvmKeys);
        UNIT_ASSERT_STRINGS_EQUAL("1:CukECuQECP0EEAAa2gQwggJWAoGBAIPYJ9-xnotYeJU3r6aX2EzFpViZ6DlLpkdPngQn2DBl4gHIsgsWZHd9Hf44DIN0Up6vS3LRJQm9aaD-4k02fPmefk1pNoJXzaOkxJJwM3QhHl6t8Yiery-6aOT4G0RVf_4kGnszOZK97a4_DaHZ1PuiySMBrtd_HiAio_O_6VJVAkEAnkxylblCyD8k-Tv5qfnHoK8DzlSFpkP9YGs7efk0smrVQTPSi2JWqiaUN--J9qJJwMeR_YJvvJyRxYqVf7zhKwJBANU331pqHm3p5s4NtJVzqfETSnOlpdDBNuLKlQVzNaZ3XHWAH_fX4nK8PJd5RT4M9rdfRYG3om49LQNjrFrl2n8CQDn3AVD2h5zIOzzCJ4QdMhkYI7iE4x4Ifp1nGe6zZ6l9v-y3iheKiLc8rX2Ig5VvU0P_VTtGsg6sURdReJKMBEoCQBqm--tNQ829PNnBtpKudT4iaU50tLoYJtxZUqCuZrTO646wA_76_E5Xh5LvKKfBntbr6LA29E3HpaBsdYtcu1ACQBPJjlK3KFkH5J8nfzU_OPQV4HnKkLTIf6wNZ28_JpZNWqgmelFsStVE0ob98T7USTgY8j-wTfeTkjixUq_3nCUCQQCWX0iz7ceFx5OUd1ZCptr3kdeY5UezZJMgIfb3fzDTSTxHTEEfkKucHHDqVz1FgdDpNP20SJVUdJw8Muw7skCEAkEA0dAtXLSrbOQr6N0sc7O-Ehc31kSFbA4Ir8vm9VhBfl-yjYkZgsQBX0cb7veTc_2epn27s9sqPSdNRkdRnTnEWyABEAA",
                                  (*c.PrivateBbKeys)[162]);
        UNIT_ASSERT_STRINGS_EQUAL("1:CucECuIECP8EEAAa2AQwggJUAoGBAJIynTEbmnKurdxAnkN0t-9cAacBjX1H2h2HGmRA3HYfLV1RId7qJaneBLUzaljMUBqaKGxTDp3hK6eZITlEn8gXFpnUiaDv9dt8GUTt0x_VKiMbt-K9GOhYPqw6hW-dyRD_FhFDYXw3LVbWvgvQ9p3IQfViY3q7bTVkb4XcchSlAkEAyE0xn0CbQtMEvw9t-Bs8CFIAo8YcbZCIGanN0N9-lJlQufQmD8rWa80_81SjlwtHgMd1DURsMDdvw6Exit-zqwJBALrZ8nTF03OjA1LuA2MfLMjyHbGh15KOHpcoWJIghqSIIvwakAC-7_ypttwEDITmTcQsD5xgufapvltzW-v0CO8CQCVc-9eqnbY14_xwZTeZEBGeOhQClgGeR22GP29LURFyIoHazFHDD7SM54KdLOwzzez8Ggb2XHYsW1O6bPcx_94CQBdbPk6Yum50YGpdwGxj5ZkeQ7Y0OvJRw9LlCxJEENSRBF-DUgAX3f-VNtuAgZCcybiFgfOMFz7VN8tua31-gR4CQBkJpjPoE2haYJfh7b8DZ4EKQBR4w42yEQM1Obob79KTKhc-hMH5Ws15p_5qlHLhaPAY7qGojYYG7fh0JjFb9nUCQDtReDs-sQdqSyspv7r7l3FoWlwhhF216c9wu6ecIVevE7KGlTS9i-rKrWmGCnpmGUND1MnlNwgz1p56iCXab7kCQBNd8xLx5-QAZSI4L2kRPh9n_ZYVNPXiswFo0NLg3K69P5wLTbPqx59t5bGBmQS1usokiPdrP1PgqA4FVY_YqIAgARAC",
                                  (*c.PrivateBbKeys)[164]);
        UNIT_ASSERT_STRINGS_EQUAL("1:CukECuQECIEFEAAa2gQwggJWAoGBANxxCZ041vsMXKCIw15Ckv3b1xntSzOaoSe0G_WLcIe6wU9wfnTYgksLxmIJlCkC4ObLsXcxyuv479tbezotw8osw20gGrh9_JwwhRBEgtQ9fVMbYBQaNIoQsNChf_bRf6a9UrnSFFfss2U7SvvQqdzV8-JbQGEnaleV85mbe-41AkEA30j2RUkD8-HfqK8kuDULo2OhD8wMFAuQSpVW5gFMajrOav--JWVvLE3Dx04qwcmLEm7x1-I4PtO133JOaa378wJBAPy9bCUr_qWnFa7AeiZ31omkr8owi3TrdKz-jhEAH0iW0pnw2TzXJZH2Hr8YbT4TuZN3V49-kGOxjJU6Yjn-PzcCQASGPOKHyJLdT8eBvf37oMIjVXWUYlsFcYpIGT6FEsGT0jar4O6R2AKGdz6Nh2sre3teBzksZE69TmITEIb7YT4CQB-XrYSlf9S04rXYD0TO-tE0lflGEW6dbpWf0cIgA-kS2lM-Gyea5LI-w9fjDafCdzJu6vHv0gx2MZKnTEc_x-cCQBvpHsipIH58O_UV5JcGoXRsdCH5gYKBcglSqtzAKY1HWc1f98SsreWJuHjpxVg5MWJN3jr8RwfadrvuSc01v34CQQCPNlcQVDnKsWQ72MBXT6ojaLhybvmgVfQod-ZN98qz0iybXuL_1BCI-SxH6a1J1mCK2j-bo6h4fHvFqdRvW7f4AkEAkZCCLTXJSJrqQOBTJO9UyHu27dX8RolMEIC82JT-J15E1GnDHdrijIXxrwAiaCvp6CGgIULtH5olgzc7dVknwyABEAE",
                                  (*c.PrivateBbKeys)[224]);
        UNIT_ASSERT_STRINGS_EQUAL("1:CuYECuEECIMFEAAa1wQwggJTAoGATpUftLb4O1HdPEQ_IBk6Y2MYYAoP6m2BgqeiWShLaVENPRgP3J4VhwvzBg7QsignoH1kC8eFlVTzh7q4uNgNpE3zZIvF2TZj9Ooc5C2UNA7YAE8r1ux0eGyj1_d5fL8gQKF0BNGgzotVR0EES6ANXJF_Wu3RiLejdJsh5G1voN0CQQCCc_ayQrGeUsmAQrkqM7Hu6ab5YZX_80hdaIQ8viotkeX6kjN_KA6-WBGlfwXEziBBSE3W1_KN9IdisLdMvSbTAkEAmjWzJ-E_oy6G2ntCA6C6h2Zjqg4qULYASeE28oRnuFxRsTe4G_21PCj2OZKLmIuw4idTG-asUjE9MEe_VsirjwJATDX-uZ83BkWzGgrOIXpgBeSaFfDX_mDfBfTWYV2QBhsOvOsx3_3zcI902AVCx0CyKRNDN9yqNDTIn9DeaXX_RwJAE0a2ZPwn9GXQ209oQHQXUOzMdUHFShbACTwm3lCM9wuKNib3A3-2p4UexzJRcxF2HETqY3zVikYnpgj36tkVcgJAEE5-1khWM8pZMAhXJUZ2Pd003ywyv_5pC60Qh5fFRbI8v1JGb-UB18sCNK_guJnECCkJutr-Ub6Q7FYW6Zek2gJAIwJW9jaNegg_KuuFPYsRSfDOsIAVYpOPb30Nn_qZBuuknNmQksQ2kSjUGxeQWG8d8lGA-bnSvxu-u3fHi2GTKgJAZJ0rJnPrDjmqoMtBnCXSRcVEUlFd8hgtUwZHvbh6kDpXcVSfvOqt_n4VXHy5SNgAzYS44EyiFpZUFNn1t0EJxyABEAM",
                                  (*c.PrivateBbKeys)[168]);
        UNIT_ASSERT_STRINGS_EQUAL("1:CucECuIECIUFEAAa2AQwggJUAoGBAOhSI4TZIoUtFqM0OyGfGzUzSHXJWDDf_in3hN5PMC6UmsgkeLjqiWFx4pFcQ5UYWs8z-xb1IRGWwSmfo20gg93RTein6yiSvci634dNDiJHNQrU1LxNI7iTG6pQd1xagO2-8MFB_xqflD_zUzDwnbtSJs3qpj9HlvnAPi98IpPVAkEA74tuVqXbMKxrMiTV-txPu91QZOuo8ZZ-LXVD5vz6FEWZQnxEmSGTjKFVaPLfF9hxN5PHmbJXPILwZmi5ngic6wJBAPhHrAPgEL7JEwRP81JZ7o8JMC952bw3KCIrnNVT6nhcYTh-xuHsYv--SBZZ99Zhzj3LE34fVASCmBuyj39WYj8CQCC4tJjkwdgAyz0IfTqjdh44IQ8dWVb4fY0e0Vhk3kYERiHgsRtuL-hzqGLfLBoMc4zxR_3BfGL0xnhqIkyJYikCQB8I9YB8AhfZImCJ_mpLPdHhJgXvOzeG5QRFc5qqfU8LjCcP2Nw9jF_3yQLLPvrMOce5Ym_D6oCQUwN2Ue_qzEgCQB3xbcrUu2YVjWZEmr9bifd7qgyddR4yz8WuqHzfn0KIsyhPiJMkMnGUKq0eW-L7DibyePM2SueQXgzNFzPBE50CQBUEBZBdYg7thD3E5F6y_T7ACyA2Pgfcu8cKjEifTGusyVDp-my7MuJ2tjeItrPmIEAPgm1GctEREct0J-VtVMkCQC-c28ip4N-65QVL88uKXT4Wf_fWMovq5eweRys3lLnmJMPwhcJTCkO9cf7825ZgFf-dm04BsELpidfO0BlBpWsgARAE",
                                  (*c.PrivateBbKeys)[170]);
        UNIT_ASSERT_STRINGS_EQUAL("1:CukECuQECP0EEAAa2gQwggJWAoGBAIPYJ9-xnotYeJU3r6aX2EzFpViZ6DlLpkdPngQn2DBl4gHIsgsWZHd9Hf44DIN0Up6vS3LRJQm9aaD-4k02fPmefk1pNoJXzaOkxJJwM3QhHl6t8Yiery-6aOT4G0RVf_4kGnszOZK97a4_DaHZ1PuiySMBrtd_HiAio_O_6VJVAkEAnkxylblCyD8k-Tv5qfnHoK8DzlSFpkP9YGs7efk0smrVQTPSi2JWqiaUN--J9qJJwMeR_YJvvJyRxYqVf7zhKwJBANU331pqHm3p5s4NtJVzqfETSnOlpdDBNuLKlQVzNaZ3XHWAH_fX4nK8PJd5RT4M9rdfRYG3om49LQNjrFrl2n8CQDn3AVD2h5zIOzzCJ4QdMhkYI7iE4x4Ifp1nGe6zZ6l9v-y3iheKiLc8rX2Ig5VvU0P_VTtGsg6sURdReJKMBEoCQBqm--tNQ829PNnBtpKudT4iaU50tLoYJtxZUqCuZrTO646wA_76_E5Xh5LvKKfBntbr6LA29E3HpaBsdYtcu1ACQBPJjlK3KFkH5J8nfzU_OPQV4HnKkLTIf6wNZ28_JpZNWqgmelFsStVE0ob98T7USTgY8j-wTfeTkjixUq_3nCUCQQCWX0iz7ceFx5OUd1ZCptr3kdeY5UezZJMgIfb3fzDTSTxHTEEfkKucHHDqVz1FgdDpNP20SJVUdJw8Muw7skCEAkEA0dAtXLSrbOQr6N0sc7O-Ehc31kSFbA4Ir8vm9VhBfl-yjYkZgsQBX0cb7veTc_2epn27s9sqPSdNRkdRnTnEWyABEAA",
                                  (*c.PrivateBbKeys)[188]);
        UNIT_ASSERT_STRINGS_EQUAL("1:CucECuIECIUFEAAa2AQwggJUAoGBAOhSI4TZIoUtFqM0OyGfGzUzSHXJWDDf_in3hN5PMC6UmsgkeLjqiWFx4pFcQ5UYWs8z-xb1IRGWwSmfo20gg93RTein6yiSvci634dNDiJHNQrU1LxNI7iTG6pQd1xagO2-8MFB_xqflD_zUzDwnbtSJs3qpj9HlvnAPi98IpPVAkEA74tuVqXbMKxrMiTV-txPu91QZOuo8ZZ-LXVD5vz6FEWZQnxEmSGTjKFVaPLfF9hxN5PHmbJXPILwZmi5ngic6wJBAPhHrAPgEL7JEwRP81JZ7o8JMC952bw3KCIrnNVT6nhcYTh-xuHsYv--SBZZ99Zhzj3LE34fVASCmBuyj39WYj8CQCC4tJjkwdgAyz0IfTqjdh44IQ8dWVb4fY0e0Vhk3kYERiHgsRtuL-hzqGLfLBoMc4zxR_3BfGL0xnhqIkyJYikCQB8I9YB8AhfZImCJ_mpLPdHhJgXvOzeG5QRFc5qqfU8LjCcP2Nw9jF_3yQLLPvrMOce5Ym_D6oCQUwN2Ue_qzEgCQB3xbcrUu2YVjWZEmr9bifd7qgyddR4yz8WuqHzfn0KIsyhPiJMkMnGUKq0eW-L7DibyePM2SueQXgzNFzPBE50CQBUEBZBdYg7thD3E5F6y_T7ACyA2Pgfcu8cKjEifTGusyVDp-my7MuJ2tjeItrPmIEAPgm1GctEREct0J-VtVMkCQC-c28ip4N-65QVL88uKXT4Wf_fWMovq5eweRys3lLnmJMPwhcJTCkO9cf7825ZgFf-dm04BsELpidfO0BlBpWsgARAE",
                                  (*c.PrivateBbKeys)[190]);
        UNIT_ASSERT_STRINGS_EQUAL("",
                                  (*c.PrivateBbKeys)[172]);

        UNIT_ASSERT_EQUAL(647, c.TvmKey->GetId());
    }

    Y_UNIT_TEST(checkKey) {
        TCryptoKey key(1);

        UNIT_ASSERT_EXCEPTION_CONTAINS(key.ThrowIfInvalid(100500),
                                       yexception,
                                       "public key is empty. key_id=1. client_id=100500");

        key.PublicKey = "kek";
        UNIT_ASSERT_EXCEPTION_CONTAINS(key.ThrowIfInvalid(100500),
                                       yexception,
                                       "private key is empty. key_id=1. client_id=100500");

        key.PrivateKey = "kek";
        UNIT_ASSERT_EXCEPTION_CONTAINS(key.ThrowIfInvalid(100500),
                                       yexception,
                                       "Public key is malformed: kek. key_id=1. client_id=100500");

        NTvmAuth::NRw::TKeyPair keys = NTvmAuth::NRw::GenKeyPair(2048);
        key.PublicKey = keys.Public;
        UNIT_ASSERT_EXCEPTION_CONTAINS(key.ThrowIfInvalid(100500),
                                       yexception,
                                       "Private key is malformed. key_id=1. client_id=100500");

        key.PrivateKey = keys.Private;
        UNIT_ASSERT_NO_EXCEPTION(key.ThrowIfInvalid(100500));

        key.PublicKey = NTvmAuth::NRw::GenKeyPair(2048).Public;
        UNIT_ASSERT_EXCEPTION_CONTAINS(key.ThrowIfInvalid(100500),
                                       yexception,
                                       "public/private key mismatch. key_id=1. client_id=100500");
    }
}
