#include <passport/infra/daemons/tvmapi/src/pregeneration/disk_cache.h>
#include <passport/infra/daemons/tvmapi/src/pregeneration/generator.h>
#include <passport/infra/daemons/tvmapi/src/pregeneration/keys.h>
#include <passport/infra/daemons/tvmapi/src/pregeneration/storage.h>
#include <passport/infra/daemons/tvmapi/src/proto/pregen_raw_list.pb.h>

#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/tvmauth/unittest.h>

using namespace NPassport::NTvm;
using namespace NPassport::NTvm::NPregen;

Y_UNIT_TEST_SUITE(Pregen) {
    Y_UNIT_TEST(Stor) {
        TKey k1{123, 456};
        TKey k2{1234, 456};
        TStorage s(std::make_shared<TStorage::TData>(TStorage::TData{{k1, "ololo"}}));
        TString r1 = s.GetTicket(k1);
        UNIT_ASSERT(r1);
        UNIT_ASSERT_STRINGS_EQUAL("ololo", r1);
        TString r2 = s.GetTicket(k2);
        UNIT_ASSERT(!r2);
    }

    Y_UNIT_TEST(Gen) {
        TKey k1{123, 456};
        TKeyList::TData keys{
            {k1,
             std::make_unique<std::atomic<time_t>>(time(nullptr))},
            {TKey{1234, 456},
             std::make_unique<std::atomic<time_t>>(100)},
        };

        TRwPrivateKeyPtr pk = std::make_shared<NTvmAuth::NRw::TRwPrivateKey>(
            NPassport::NUtils::Base64url2bin("MIICVQKBgQC4a4oW39xKYw0EtrAkB2s6BYDdJwWxPXnrJ5xU380BC129oJyVPXp5lf2g6mhyo4LSyD6QoJ-NJR65-ZOprx-TnXnciOnAsxPjfhhad5wKtDQycjHtH4bSEQ__wm09LpXL5HqMOubm11n8OnQQtoEf9-K3THvy_0xkSzFow1C2zQJBANJufhSF9qTZnFlA73M3Mofzhf2AqVqe2L2uN5S9mQyhx6o8zzBm7pGiMiIea9dmOhLyS_BihHkmq4dIw40a25MCQQDgWxGhgsLCiR8R7qd7ixC669xRXyimJz5QYpWKeXhZH4fUjLIbgRRQVhKezMunkAs81yPSVwFyzla8kFTYA2AfAkEAigT7h01LzA_rL9xSyd_I17dSZkVisp7vdyxnrjD1iZqX2IPF9RTrEi7sboCaaP1hS2-G1vs1he3QNdcoTyEGegJAHAtiNDBYWFEj4j3U73FiF117iivlFMTnygxSsU8vCyPw-pGWQ3AiigrCU9mZdPIBZ5rkekrgLlnK15IKmwBsBAJAGk3PwpC-1Jsziygd7mbmUP5wv7AVK1PbF7XG8pezIZQ49UeZ5gzd0jRGREPNeuzHQl5JfgxQjyTVcOkYcaNbcgJATBRJcYoDdz5Y4Wq3Aui8cu8eMFe7iL1Q63twjW47OBjkmLKYYckGryfPHr9IISSazZHAnphq3TkCDZWRYhoMDQJASwsbzFCz7PcNqmQzPBi3foWh7PsYGgpwYINuBS4VnWhIlSaUz3y4HYIyBywVYkIfHfatfRtoT3pNrcosXH0BYg"),
            16);

        TGenerator gen(TGenerator::TSettings{
            .KeyTtl = 100,
            .TicketTtl = 100500,
        });

        NTvmAuth::TServiceContext ctx = NTvmAuth::TServiceContext::CheckingFactory(1, NTvmAuth::NUnittest::TVMKNIFE_PUBLIC_KEYS);
        TStorage::TDataPtr r2 = gen.GenAll(keys, pk, ctx);
        UNIT_ASSERT_VALUES_EQUAL(1, r2->size());
        UNIT_ASSERT_VALUES_EQUAL(k1, r2->begin()->first);

        pk = std::make_shared<NTvmAuth::NRw::TRwPrivateKey>(
            NPassport::NUtils::Base64url2bin("MIICVQKBgQC4a4oW39xKYw0EtrAkB2s6BYDdJwWxPXnrJ5xU380BC129oJyVPXp5lf2g6mhyo4LSyD6QoJ-NJR65-ZOprx-TnXnciOnAsxPjfhhad5wKtDQycjHtH4bSEQ__wm09LpXL5HqMOubm11n8OnQQtoEf9-K3THvy_0xkSzFow1C2zQJBANJufhSF9qTZnFlA73M3Mofzhf2AqVqe2L2uN5S9mQyhx6o8zzBm7pGiMiIea9dmOhLyS_BihHkmq4dIw40a25MCQQDgWxGhgsLCiR8R7qd7ixC669xRXyimJz5QYpWKeXhZH4fUjLIbgRRQVhKezMunkAs81yPSVwFyzla8kFTYA2AfAkEAigT7h01LzA_rL9xSyd_I17dSZkVisp7vdyxnrjD1iZqX2IPF9RTrEi7sboCaaP1hS2-G1vs1he3QNdcoTyEGegJAHAtiNDBYWFEj4j3U73FiF117iivlFMTnygxSsU8vCyPw-pGWQ3AiigrCU9mZdPIBZ5rkekrgLlnK15IKmwBsBAJAGk3PwpC-1Jsziygd7mbmUP5wv7AVK1PbF7XG8pezIZQ49UeZ5gzd0jRGREPNeuzHQl5JfgxQjyTVcOkYcaNbcgJATBRJcYoDdz5Y4Wq3Aui8cu8eMFe7iL1Q63twjW47OBjkmLKYYckGryfPHr9IISSazZHAnphq3TkCDZWRYhoMDQJASwsbzFCz7PcNqmQzPBi3foWh7PsYGgpwYINuBS4VnWhIlSaUz3y4HYIyBywVYkIfHfatfRtoT3pNrcosXH0BYg"),
            42);

        UNIT_ASSERT_EXCEPTION_CONTAINS(gen.GenOne(keys.begin()->first, *pk, ctx, 100500),
                                       yexception,
                                       "Failed to generate service ticket: totally unexpected scenario");
    }

    Y_UNIT_TEST(serialize) {
        const time_t now = time(nullptr);
        TString serialized;

        {
            TDiskCache dc = TDiskCache::FromMemory("", 100, 100);

            TKey k1{123, 456};
            TKeyList::TData keys{
                {k1,
                 std::make_unique<std::atomic<time_t>>(now)},
                {TKey{1234, 456},
                 std::make_unique<std::atomic<time_t>>(100)},
            };

            TString ticket1 = "3:serv:CBAQ__________9_IgUIexDIAw:HIeigTZltAcImAqrO2uqxVQj1rL_b_yHnxH4y1C7-iFq53p7S6aXNTsOj9DXLlRFPlhLYR70mzGiLwHmZYmV_Z3jHL5shnj0a7mO8DPXjmjyelc3yp1UBV4TTGw9ESP0lugfPs7HGIkcGl-UMiwO5lO58NFKRum1NaUIW7VBP0E";
            TStorage::TData tickets{
                {k1, ticket1},
            };

            serialized = dc.ToArray(keys, tickets, now + 1000);
            UNIT_ASSERT(serialized.empty());

            serialized = dc.ToArray(keys, tickets, now);
            UNIT_ASSERT(!serialized.empty());

            pregen_raw_list::Data proto;
            UNIT_ASSERT(proto.ParseFromArray(serialized.data(), serialized.size()));
            UNIT_ASSERT_VALUES_EQUAL(1, proto.row_size());
            UNIT_ASSERT_VALUES_EQUAL(123, proto.row(0).src());
            UNIT_ASSERT_VALUES_EQUAL(456, proto.row(0).dst());
            UNIT_ASSERT_VALUES_EQUAL(now, proto.row(0).last_time());
            UNIT_ASSERT_VALUES_EQUAL(ticket1, proto.row(0).ticket());
        }

        {
            TDiskCache dc2 = TDiskCache::FromMemory(serialized, 100, 100);
            TKeyList::TData keys = dc2.ExtractKeys();
            UNIT_ASSERT_VALUES_EQUAL(1, keys.size());

            TStorage::TData tickets = dc2.ExtractTickets();
            UNIT_ASSERT_VALUES_EQUAL(1, tickets.size());
        }

        {
            TDiskCache dc2 = TDiskCache::FromMemory(serialized, 100, 100, now + 1000);
            TKeyList::TData keys = dc2.ExtractKeys();
            UNIT_ASSERT_VALUES_EQUAL(0, keys.size());

            TStorage::TData tickets = dc2.ExtractTickets();
            UNIT_ASSERT_VALUES_EQUAL(0, tickets.size());
        }
    }

    Y_UNIT_TEST(fromCache) {
        const TString raw = NPassport::NUtils::Base64ToBin(
            "CiAI0gkQyAMaBnNjb3BlMRoGc2NvcGUyGgZzY29wZTMgZAojCHsQyAMaBnNjb3BlMRoGc2NvcGUyGgZzY29wZTMgzs2kygU=");

        TKeyList::TData k1 = TDiskCache::FromMemory(raw, 0, 0).ExtractKeys();
        UNIT_ASSERT(k1.empty());
        TStorage::TData t1 = TDiskCache::FromMemory(raw, 0, 0).ExtractTickets();
        UNIT_ASSERT(t1.empty());

        TKeyList::TData r2 = TDiskCache::FromMemory(raw, 1000000000UL, 1000000000UL).ExtractKeys();
        UNIT_ASSERT_EQUAL(1, r2.size());
        TKey k{123, 456};
        UNIT_ASSERT_EQUAL(k, r2.begin()->first);

        TStorage::TData t2 = TDiskCache::FromMemory(raw, 1000000000UL, 1000000000UL).ExtractTickets();
        UNIT_ASSERT(t2.empty());
    }
}

template <>
void Out<TKey>(IOutputStream& o, const TKey& value) {
    o << "src=" << value.Src << ";dst=" << value.Dst;
}
