#include <passport/infra/daemons/tvmapi/src/runtime_context/public_keys_builder.h>

#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/tvmauth/unittest.h>
#include <library/cpp/tvmauth/src/parser.h>

using namespace NPassport::NTvm;
using namespace NPassport::NUtils;

Y_UNIT_TEST_SUITE(PublicKeys) {
    Y_UNIT_TEST(Ok) {
        TPublicKeysBuilder pkb;
        UNIT_ASSERT_STRINGS_EQUAL("1:", pkb.SerializeV1());

        pkb.AddBbKey(1,
                     100501,
                     tvm_keys::Prod,
                     Base64url2bin("MIIBBAKCAQBwsRd4frsVARIVSfj_vCdfvA3Q9SsGhSybdBDhbm8L6rPqxdoSNLCdNXzDWj7Ppf0o8uWHMxC-5Lfw0I18ri68nhm9-ndixcnbn6ti1uetgkc28eiEP6Q8ILD_JmkynbUl1aKDNAa5XsK2vFSEX402uydRomsTn46kRY23hfqcIi0ohh5VxIrpclRsRZus0JFu-RJzhqTbKYV4y4dglWPGHh5BuTv9k_Oh0_Ra8Xp5Rith5vjaKZUQ5Hyh9UtBYTkNWdvXP9OpmbiLVeRLuMzBm4HEFHDwMZ1h6LSVP-wB_spJPaMLTn3Q3JIHe-wGBYRWzU51RRYDqv4O_H12w5C1"));
        pkb.AddBbKey(2,
                     100502,
                     tvm_keys::ProdYateam,
                     Base64url2bin("MIIBBQKCAQEA4RATOfumLD1n6ICrW5biaAl9VldinczmkNPjpUWwc3gs8PnkCrtdnPFmpBwW3gjHdSNU1OuEg5A6K1o1xiGv9sU-jd88zQBOdK6E2zwnJnkK6bNusKE2H2CLqg3aMWCmTa9JbzSy1uO7wa-xCqqNUuCko-2lyv12HhL1ICIH951SHDa4qO1U5xZhhlUAnqWi9R4tYDeMiF41WdOjwT2fg8UkbusThmxa3yjCXjD7OyjshPtukN8Tl3UyGtV_s2CLnE3f28VAi-AVW8FtgL22xbGhuyEplXRrtF1E5oV7NSqxH1FS0SYROA8ffYQGV5tfx5WDFHiXDEP6BzoVfeBDRQ=="));
        pkb.AddBbKey(3,
                     100503,
                     tvm_keys::Test,
                     Base64url2bin("MIGDAoGAX23ZgkYAmRFEWrp9aGLebVMVbVQ4TR_pmt9iEcCSmoaUqWHRBV95M0-l4mGLvnFfMJ7qhF5FSb7QNuoM2FNKELu4ZS_Ug1idEFBYfoT7kVzletsMVK4ZDDYRiM18fL8d58clfFAoCo-_EEMowqQeBXnxa0zqsLyNGL2x1f-KDY0="));
        pkb.AddBbKey(4,
                     100504,
                     tvm_keys::TestYateam,
                     Base64url2bin("MIGDAoGAYHh3p4sZQG_5DoQ8t6ELhL7K4TYcH7-sntaR5jDKn7Eg-iU-t349CZ7a60cHhmClcci784WSwN7_Rs-BmeI1FJVFuXtbj0OBLlhreQx-tgHgOcXkW1rm2fWhXoTDFgPgk42kVN6c4kAs8ZM9rDFcR-if-l6Ic5IG5Ay7f9Wf6XU"));
        pkb.AddBbKey(5,
                     100505,
                     tvm_keys::Stress,
                     Base64url2bin("MIGEAoGBAJMkwoh-Z95mlojtD2uexzb2B1ZrArtOelUpEfCzWJjsRSVhE55Vwx0DASpUzp_wFUIosaDNDsZqv-67F9QYrCoRDOimkOtrDgXvknjrj7sPl_r-glC4YgEdnGpsw420uMEJdFSFBbmzMEuZND9Hepolvm9_6HQA9l-RiGrVxO21"));

        pkb.AddTvmKey(6,
                      100506,
                      Base64url2bin("MIGEAoGBAKhmQUab2gtOqN5dkCrqPlLqLsrbRXSU10EEup-YUEhG5ORjGGzi9RidaNmspEB3lunr-nLRyyYWamX_liHRv7al2j8Tk80_ssZHMrIpKIV13LKQorNU5rMfxhiV0MB8TxnkehL17jxqfkmw-DSOv5M-7fgOcmrfK_Zrp2qIov3N"));

        pkb.AddBbKey(11,
                     100501,
                     tvm_keys::Prod,
                     // priv: MIICUwKBgG-8ZUf8zYfyrHG_3jJoNhRNZtGcnW6Ihhzn0ohCV8WKYw-FJLvHa2pRrb6EjnkmIgjRfKboY2TAuHJzXG1YKGlW4h1Kdu-21uu3jB4MwsepPA04XqoCxUoYpPViHzJ_6pv4NKaJyC8VzviYFCb_qABjWJCx_fdhfznvwKWEoFYdAkEAqAx2xBrj46eFBQVlJUhWpmGt2lMBenVBzBZyr37diLFu0lJWP1iw_O9scMlg1EhVeEZ7ThoYJxw2za2wtvJ42wJBAKo3BbC2sNRyit23n8A54gNRCA8JFrAYSj8YYZnkVE1DiO71VgUnX4WVqQiCg8Ww3kFYlPkb-UcKIeQlX7ZgAmcCQBbqfprF7CYY1UGHk-Sig1fHm56G9s_P5D8D4UFXBsgRzdWvhGFD2xkgrd2Y-dPtozxtBFfSh4-WBQM9KcRkJnwCQBVG4LYW1hqOUVu28_gHPEBqIQHhItYDCUfjDDM8iomocR3eqsCk6_CytSEQUHi2G8grEp8jfyjhRDyEq_bMAE0CQBUBjtiDXHx08KCgrKSpCtTMNbtKYC9OqDmCzlXv27EWLdpKSsfrFh-d7Y4ZLBqJCq8Iz2nDQwTjhtm1thbeTxsCQDM0f9paa8Q8gwNYkUYe0DDtzL9h7Eg5rC_ecrW81MUcWLSZuzNt_ZO6r4Aya3CNTM79km4bnBn1NJedwVSTDrYCQDnAVWh2oObpjCcWXUBWAQmJVtfz4ldKyT6Oa0vkY52wOYR_GG2jLgkg832nyAWVGD-HDEiMDzoT7lIyC-8f_q4
                     Base64url2bin("MIGDAoGAb7xlR_zNh_Kscb_eMmg2FE1m0ZydboiGHOfSiEJXxYpjD4Uku8dralGtvoSOeSYiCNF8puhjZMC4cnNcbVgoaVbiHUp277bW67eMHgzCx6k8DTheqgLFShik9WIfMn_qm_g0ponILxXO-JgUJv-oAGNYkLH992F_Oe_ApYSgVh0"));
        pkb.AddBbKey(12,
                     100502,
                     tvm_keys::ProdYateam,
                     // priv: MIICVAKBgQCS7MVx3lMm7uVhZh7aFAsV9RYgiP3UG9BAtr9OGWfhi0YI7yAbUomomb5iWYk5ZAbQHsf_lFWHFTFX0qmYWewNWTNQUo6wIFpgZdzXbE9WKhrBSUDCnmqndBC1nLH-x73tgigvO4uvjpSQtwueQiun4mjjpDNiL9AerjlZObwxDQJBAN8g42JCDRQL-nU-Dgumi2A6KfCWHYZ9crwWeDJInZo00yy8t165SW_NQ6My3qGJPhX9s_-3eohQL3qGBwK5xRMCQQCoke0IM2NIbwEHUSV3L_857YkjFSO7ARtLsc78k0D3357JHroLRP2LfTY2HwZHxI3BsDmcLMx-n9u5dJOniJVfAkAYgrnGTPkUziiJ6lLZcKHKXmqpWdmDdxN6YzSAICh6p2HygGNXkMhsODRB-v12ZdUtVRxOpwfG6lssJC17-M6rAkAVEj2hBmxpDeAg6iSu5f_nPbEkYqR3YCNpdjnfkmge-_PZI9dBaJ-xb6bGw-DI-JG4NgczhZmP0_t3LpJ08RKsAkAb5BxsSEGigX9Op8HBdNFsB0U-EsOwz65Xgs8GSROzRppll5br1ykt-ah0ZlvUMSfCv7Z_9u9RCgXvUMDgVziiAkBilejZzAYup1jQyEw1O9fNzQLLL44fiBI67EvCSjs0UTW4j2MFUzd5sF7y24eYxERuRjaZLFiG6qbNOqaYR4aYAkB6nsqmxdz2BNCJWeuk7TmNjc4a76qHc8mIJi2HGmvnjERyPCRwg9s18k3uls0LMp01qZx3Ky9OyVXt0Sfw9sjI
                     Base64url2bin("MIGEAoGBAJLsxXHeUybu5WFmHtoUCxX1FiCI_dQb0EC2v04ZZ-GLRgjvIBtSiaiZvmJZiTlkBtAex_-UVYcVMVfSqZhZ7A1ZM1BSjrAgWmBl3NdsT1YqGsFJQMKeaqd0ELWcsf7Hve2CKC87i6-OlJC3C55CK6fiaOOkM2Iv0B6uOVk5vDEN"));
        pkb.AddBbKey(13,
                     100503,
                     tvm_keys::Test,
                     // priv: MIICVAKBgQCFM28TpJ13_6zbde_4q_CZVr67ZwlrEHdTAJ5eZoAK72rw7SngWB84ufwJOwx5S87gOmBPFte-RcSxSqTdoZM_pfjO5bVEnfDdQbyUYSuLeln31nwv9cD8qAG3kjdmG9Y8oLWSUPMBqFx3CUFF4jIrd3hTcspcLCRY9YMwEeCQnQJBAKV2lHRBYjFkrP0YKjh2ORfMWa_ZVsyK-9ZJzVf-SZdnmQNGpL0cKcf3lUOXeMxOTCC3HfwChyZU91Cv5GdVmwMCQQDOFa6pInktP9QbA1vnZgl-PLzAtJBrYm1JLxNEf2-yq1INmkXXjk7-Zy3mHBph3AXkU2J1fQt8YhMQ020hVIPfAkB4W6hR7PH0YVgJhE6Hnj3U_8bEmdvPD7URGtvdkqNM-cpR6WCHayt2WrRaCRslVT0344sTHa3kyyPyt2rFrCq8AkAZwrXVJE8lp_qDYGt87MEvx5eYFpINbE2pJeJoj-32VWpBs0i68cnfzOW8w4NMO4C8imxOr6FvjEJiGm2kKpB8AkAUrtKOiCxGLJWfowVHDsci-Ys1-yrZkV96yTmq_8ky7PMgaNSXo4U4_vKocu8ZicmEFuO_gFDkyp7qFfyM6rNgAkBOjawH30QJSX34KKqyNDcCia9GyNyai-67TIS7DVvEw_uKNPm6vNzlaIGI_kNo0sHNCy0QtQBKo2jzsC230BVdAkA8m8iZhL-98cA9VatBWTKV9VCMAjlnqtJuxDC9B-22ybhoza_rWnjWyvE0oJFzifFjeT1Gw6gGpN7_EDBRBHVE
                     Base64url2bin("MIGEAoGBAIUzbxOknXf_rNt17_ir8JlWvrtnCWsQd1MAnl5mgArvavDtKeBYHzi5_Ak7DHlLzuA6YE8W175FxLFKpN2hkz-l-M7ltUSd8N1BvJRhK4t6WffWfC_1wPyoAbeSN2Yb1jygtZJQ8wGoXHcJQUXiMit3eFNyylwsJFj1gzAR4JCd"));
        pkb.AddBbKey(14,
                     100504,
                     tvm_keys::TestYateam,
                     // priv: MIICUwKBgFMcbEpl9ukVR6AO_R6sMyiU11I8b8MBSUCEC15iKsrVO8v_m47_TRRjWPYtQ9eZ7o1ocNJHaGUU7qqInFqtFaVnIceP6NmCsXhjs3MLrWPS8IRAy4Zf4FKmGOx3N9O2vemjUygZ9vUiSkULdVrecinRaT8JQ5RG4bUMY04XGIwFAkEAhqoUENwvyMLST7AofTdCAd1fcb2H72qWmQy3ir_FsAR9fUUNslMln-WyBmy9eeMzNpjzejRgLzqtob9u931-YwJBAJ3-9a22nOM3OoA8L0z9210h5JwnMUi5X_pFHoshKY7gvgDI_wz3rj2CIltemG-HYLSt3Tnw8AOfqQJKZGm1xHcCQAXO9oT-aWpMS8QM6k7WzR_AguMY48KPVzOlb-SdbtxTPiVaYZ9oPoYrumcB170KjS4_fj-5aiTqrP4cPgycDTgCQBO_3rW205xm51AHhemfu2ukPJOE5ikXK_9Io9FkJTHcF8AZH-Ge9cewREtr0w3w7BaVu6c-HgBz9SBJTI02uI8CQBDVQoIbhfkYWkn2BQ-m6EA7q-43sP3tUtMhlvFX-LYAj6-oobZKZLP8tkDNl688ZmbTHm9GjAXnVbQ37d7vr8wCQEmwF8MCrJ1cXQ6Eb9bhaWFQ-ivfftXjqC4-1SRjPfdwYPwWFGLpqv256ADkeiD2noYKCCbmqvFFJ96qtKuW_ekCQGn6iKuVXJd9n-hhSIkTxW28VPnfhDA3H8HWDH_RTpRMlLc8IytkIo2cjNuWO-uAY2UZp98WDKCzFdchG0213v0
                     Base64url2bin("MIGDAoGAUxxsSmX26RVHoA79HqwzKJTXUjxvwwFJQIQLXmIqytU7y_-bjv9NFGNY9i1D15nujWhw0kdoZRTuqoicWq0VpWchx4_o2YKxeGOzcwutY9LwhEDLhl_gUqYY7Hc307a96aNTKBn29SJKRQt1Wt5yKdFpPwlDlEbhtQxjThcYjAU"));
        pkb.AddBbKey(15,
                     100505,
                     tvm_keys::Stress,
                     // priv: MIICUwKBgGpCkW-NR3li8GlRvqpq2YZGSIgm_PTyDI2Zwfw69grsBmPpVFW48Vw7xoMN35zcrojEpialB_uQzlpLYOvsMl634CRIuj-n1QE3-gaZTTTE8mg-AR4mcxnTKThPnRQpbuOlYAnriwiasWiQEMbGjq_HmWioYYxFo9USlklQn4-9AkEAyZBz_sQPZW5HSxGmbspszDzYpbitf7R_5E5ZM1zYlq_5_uElqNHh8AylODpshBAi9JiMi0KTMeUy716_hpaD6wJBAIb1ERyyO1sZ-eS7pJE5kfyOyPkNWClgFrsTQ5q7c4FfG1IqS5z2HqYxG7TvN_3QqonTL1IhiIFCLPJo5BRI2PcCQEhzOLNxP1z1o9GbuclKvWpCZP9j1Z-Y7slbLbu2MnvHAmEY0vdTmsWbSmMMxi4OgXm-o09Sj8xVELPUd6lH6cACQBDeoiOWR2tjPzyXdJInMj-R2R8hqwUsAtdiaHNXbnAr42pFSXOew9TGI3ad5v-6FVE6ZepEMRAoRZ5NHIKJGx8CQBkyDn_YgeytyOliNM3ZTZmHmxS3Fa_2j_yJyyZrmxLV_z_cJLUaPD4BlKcHTZCCBF6TEZFoUmY8pl3r1_DS0H0CQFvrLWyeai0S-pgekBNdAQyiruBU5fachNxUnJRb3CzPv0iWKw9OmDm4Mp7zBQAskf02AyrLPaveQqT1w1vKE-gCQCgBq_XA-IqujM9UjF_PIa5z6JgBGrsXREeOFcDggfXPxbSxXj7tQdTCKvGR272l0b21MD7Q6njXhoeezWA_rh4
                     Base64url2bin("MIGDAoGAakKRb41HeWLwaVG-qmrZhkZIiCb89PIMjZnB_Dr2CuwGY-lUVbjxXDvGgw3fnNyuiMSmJqUH-5DOWktg6-wyXrfgJEi6P6fVATf6BplNNMTyaD4BHiZzGdMpOE-dFClu46VgCeuLCJqxaJAQxsaOr8eZaKhhjEWj1RKWSVCfj70"));

        pkb.AddTvmKey(16,
                      100506,
                      // priv: MIICVQKBgQC4a4oW39xKYw0EtrAkB2s6BYDdJwWxPXnrJ5xU380BC129oJyVPXp5lf2g6mhyo4LSyD6QoJ-NJR65-ZOprx-TnXnciOnAsxPjfhhad5wKtDQycjHtH4bSEQ__wm09LpXL5HqMOubm11n8OnQQtoEf9-K3THvy_0xkSzFow1C2zQJBANJufhSF9qTZnFlA73M3Mofzhf2AqVqe2L2uN5S9mQyhx6o8zzBm7pGiMiIea9dmOhLyS_BihHkmq4dIw40a25MCQQDgWxGhgsLCiR8R7qd7ixC669xRXyimJz5QYpWKeXhZH4fUjLIbgRRQVhKezMunkAs81yPSVwFyzla8kFTYA2AfAkEAigT7h01LzA_rL9xSyd_I17dSZkVisp7vdyxnrjD1iZqX2IPF9RTrEi7sboCaaP1hS2-G1vs1he3QNdcoTyEGegJAHAtiNDBYWFEj4j3U73FiF117iivlFMTnygxSsU8vCyPw-pGWQ3AiigrCU9mZdPIBZ5rkekrgLlnK15IKmwBsBAJAGk3PwpC-1Jsziygd7mbmUP5wv7AVK1PbF7XG8pezIZQ49UeZ5gzd0jRGREPNeuzHQl5JfgxQjyTVcOkYcaNbcgJATBRJcYoDdz5Y4Wq3Aui8cu8eMFe7iL1Q63twjW47OBjkmLKYYckGryfPHr9IISSazZHAnphq3TkCDZWRYhoMDQJASwsbzFCz7PcNqmQzPBi3foWh7PsYGgpwYINuBS4VnWhIlSaUz3y4HYIyBywVYkIfHfatfRtoT3pNrcosXH0BYg
                      Base64url2bin("MIGEAoGBALhrihbf3EpjDQS2sCQHazoFgN0nBbE9eesnnFTfzQELXb2gnJU9enmV_aDqaHKjgtLIPpCgn40lHrn5k6mvH5OdedyI6cCzE-N-GFp3nAq0NDJyMe0fhtIRD__CbT0ulcvkeow65ubXWfw6dBC2gR_34rdMe_L_TGRLMWjDULbN"));

        UNIT_ASSERT_STRINGS_EQUAL(NTvmAuth::NUnittest::TVMKNIFE_PUBLIC_KEYS,
                                  pkb.SerializeV1());

        UNIT_ASSERT(NTvmAuth::TParserTvmKeys::ParseStrV1(pkb.SerializeV1()));
    }

    Y_UNIT_TEST(PrivateKeys) {
        TPublicKeysBuilder pkb;
        UNIT_ASSERT_STRINGS_EQUAL("1:", pkb.SerializeV1());

        pkb.AddBbKey(11,
                     100501,
                     tvm_keys::Prod,
                     Base64url2bin("MIICUwKBgG-8ZUf8zYfyrHG_3jJoNhRNZtGcnW6Ihhzn0ohCV8WKYw-FJLvHa2pRrb6EjnkmIgjRfKboY2TAuHJzXG1YKGlW4h1Kdu-21uu3jB4MwsepPA04XqoCxUoYpPViHzJ_6pv4NKaJyC8VzviYFCb_qABjWJCx_fdhfznvwKWEoFYdAkEAqAx2xBrj46eFBQVlJUhWpmGt2lMBenVBzBZyr37diLFu0lJWP1iw_O9scMlg1EhVeEZ7ThoYJxw2za2wtvJ42wJBAKo3BbC2sNRyit23n8A54gNRCA8JFrAYSj8YYZnkVE1DiO71VgUnX4WVqQiCg8Ww3kFYlPkb-UcKIeQlX7ZgAmcCQBbqfprF7CYY1UGHk-Sig1fHm56G9s_P5D8D4UFXBsgRzdWvhGFD2xkgrd2Y-dPtozxtBFfSh4-WBQM9KcRkJnwCQBVG4LYW1hqOUVu28_gHPEBqIQHhItYDCUfjDDM8iomocR3eqsCk6_CytSEQUHi2G8grEp8jfyjhRDyEq_bMAE0CQBUBjtiDXHx08KCgrKSpCtTMNbtKYC9OqDmCzlXv27EWLdpKSsfrFh-d7Y4ZLBqJCq8Iz2nDQwTjhtm1thbeTxsCQDM0f9paa8Q8gwNYkUYe0DDtzL9h7Eg5rC_ecrW81MUcWLSZuzNt_ZO6r4Aya3CNTM79km4bnBn1NJedwVSTDrYCQDnAVWh2oObpjCcWXUBWAQmJVtfz4ldKyT6Oa0vkY52wOYR_GG2jLgkg832nyAWVGD-HDEiMDzoT7lIyC-8f_q4"));
        pkb.AddBbKey(12,
                     100502,
                     tvm_keys::ProdYateam,
                     Base64url2bin("MIICVAKBgQCS7MVx3lMm7uVhZh7aFAsV9RYgiP3UG9BAtr9OGWfhi0YI7yAbUomomb5iWYk5ZAbQHsf_lFWHFTFX0qmYWewNWTNQUo6wIFpgZdzXbE9WKhrBSUDCnmqndBC1nLH-x73tgigvO4uvjpSQtwueQiun4mjjpDNiL9AerjlZObwxDQJBAN8g42JCDRQL-nU-Dgumi2A6KfCWHYZ9crwWeDJInZo00yy8t165SW_NQ6My3qGJPhX9s_-3eohQL3qGBwK5xRMCQQCoke0IM2NIbwEHUSV3L_857YkjFSO7ARtLsc78k0D3357JHroLRP2LfTY2HwZHxI3BsDmcLMx-n9u5dJOniJVfAkAYgrnGTPkUziiJ6lLZcKHKXmqpWdmDdxN6YzSAICh6p2HygGNXkMhsODRB-v12ZdUtVRxOpwfG6lssJC17-M6rAkAVEj2hBmxpDeAg6iSu5f_nPbEkYqR3YCNpdjnfkmge-_PZI9dBaJ-xb6bGw-DI-JG4NgczhZmP0_t3LpJ08RKsAkAb5BxsSEGigX9Op8HBdNFsB0U-EsOwz65Xgs8GSROzRppll5br1ykt-ah0ZlvUMSfCv7Z_9u9RCgXvUMDgVziiAkBilejZzAYup1jQyEw1O9fNzQLLL44fiBI67EvCSjs0UTW4j2MFUzd5sF7y24eYxERuRjaZLFiG6qbNOqaYR4aYAkB6nsqmxdz2BNCJWeuk7TmNjc4a76qHc8mIJi2HGmvnjERyPCRwg9s18k3uls0LMp01qZx3Ky9OyVXt0Sfw9sjI"));
        pkb.AddBbKey(13,
                     100503,
                     tvm_keys::Test,
                     Base64url2bin("MIICVAKBgQCFM28TpJ13_6zbde_4q_CZVr67ZwlrEHdTAJ5eZoAK72rw7SngWB84ufwJOwx5S87gOmBPFte-RcSxSqTdoZM_pfjO5bVEnfDdQbyUYSuLeln31nwv9cD8qAG3kjdmG9Y8oLWSUPMBqFx3CUFF4jIrd3hTcspcLCRY9YMwEeCQnQJBAKV2lHRBYjFkrP0YKjh2ORfMWa_ZVsyK-9ZJzVf-SZdnmQNGpL0cKcf3lUOXeMxOTCC3HfwChyZU91Cv5GdVmwMCQQDOFa6pInktP9QbA1vnZgl-PLzAtJBrYm1JLxNEf2-yq1INmkXXjk7-Zy3mHBph3AXkU2J1fQt8YhMQ020hVIPfAkB4W6hR7PH0YVgJhE6Hnj3U_8bEmdvPD7URGtvdkqNM-cpR6WCHayt2WrRaCRslVT0344sTHa3kyyPyt2rFrCq8AkAZwrXVJE8lp_qDYGt87MEvx5eYFpINbE2pJeJoj-32VWpBs0i68cnfzOW8w4NMO4C8imxOr6FvjEJiGm2kKpB8AkAUrtKOiCxGLJWfowVHDsci-Ys1-yrZkV96yTmq_8ky7PMgaNSXo4U4_vKocu8ZicmEFuO_gFDkyp7qFfyM6rNgAkBOjawH30QJSX34KKqyNDcCia9GyNyai-67TIS7DVvEw_uKNPm6vNzlaIGI_kNo0sHNCy0QtQBKo2jzsC230BVdAkA8m8iZhL-98cA9VatBWTKV9VCMAjlnqtJuxDC9B-22ybhoza_rWnjWyvE0oJFzifFjeT1Gw6gGpN7_EDBRBHVE"));
        pkb.AddBbKey(14,
                     100504,
                     tvm_keys::TestYateam,
                     Base64url2bin("MIICUwKBgFMcbEpl9ukVR6AO_R6sMyiU11I8b8MBSUCEC15iKsrVO8v_m47_TRRjWPYtQ9eZ7o1ocNJHaGUU7qqInFqtFaVnIceP6NmCsXhjs3MLrWPS8IRAy4Zf4FKmGOx3N9O2vemjUygZ9vUiSkULdVrecinRaT8JQ5RG4bUMY04XGIwFAkEAhqoUENwvyMLST7AofTdCAd1fcb2H72qWmQy3ir_FsAR9fUUNslMln-WyBmy9eeMzNpjzejRgLzqtob9u931-YwJBAJ3-9a22nOM3OoA8L0z9210h5JwnMUi5X_pFHoshKY7gvgDI_wz3rj2CIltemG-HYLSt3Tnw8AOfqQJKZGm1xHcCQAXO9oT-aWpMS8QM6k7WzR_AguMY48KPVzOlb-SdbtxTPiVaYZ9oPoYrumcB170KjS4_fj-5aiTqrP4cPgycDTgCQBO_3rW205xm51AHhemfu2ukPJOE5ikXK_9Io9FkJTHcF8AZH-Ge9cewREtr0w3w7BaVu6c-HgBz9SBJTI02uI8CQBDVQoIbhfkYWkn2BQ-m6EA7q-43sP3tUtMhlvFX-LYAj6-oobZKZLP8tkDNl688ZmbTHm9GjAXnVbQ37d7vr8wCQEmwF8MCrJ1cXQ6Eb9bhaWFQ-ivfftXjqC4-1SRjPfdwYPwWFGLpqv256ADkeiD2noYKCCbmqvFFJ96qtKuW_ekCQGn6iKuVXJd9n-hhSIkTxW28VPnfhDA3H8HWDH_RTpRMlLc8IytkIo2cjNuWO-uAY2UZp98WDKCzFdchG0213v0"));
        pkb.AddBbKey(15,
                     100505,
                     tvm_keys::Stress,
                     Base64url2bin("MIICUwKBgGpCkW-NR3li8GlRvqpq2YZGSIgm_PTyDI2Zwfw69grsBmPpVFW48Vw7xoMN35zcrojEpialB_uQzlpLYOvsMl634CRIuj-n1QE3-gaZTTTE8mg-AR4mcxnTKThPnRQpbuOlYAnriwiasWiQEMbGjq_HmWioYYxFo9USlklQn4-9AkEAyZBz_sQPZW5HSxGmbspszDzYpbitf7R_5E5ZM1zYlq_5_uElqNHh8AylODpshBAi9JiMi0KTMeUy716_hpaD6wJBAIb1ERyyO1sZ-eS7pJE5kfyOyPkNWClgFrsTQ5q7c4FfG1IqS5z2HqYxG7TvN_3QqonTL1IhiIFCLPJo5BRI2PcCQEhzOLNxP1z1o9GbuclKvWpCZP9j1Z-Y7slbLbu2MnvHAmEY0vdTmsWbSmMMxi4OgXm-o09Sj8xVELPUd6lH6cACQBDeoiOWR2tjPzyXdJInMj-R2R8hqwUsAtdiaHNXbnAr42pFSXOew9TGI3ad5v-6FVE6ZepEMRAoRZ5NHIKJGx8CQBkyDn_YgeytyOliNM3ZTZmHmxS3Fa_2j_yJyyZrmxLV_z_cJLUaPD4BlKcHTZCCBF6TEZFoUmY8pl3r1_DS0H0CQFvrLWyeai0S-pgekBNdAQyiruBU5fachNxUnJRb3CzPv0iWKw9OmDm4Mp7zBQAskf02AyrLPaveQqT1w1vKE-gCQCgBq_XA-IqujM9UjF_PIa5z6JgBGrsXREeOFcDggfXPxbSxXj7tQdTCKvGR272l0b21MD7Q6njXhoeezWA_rh4"));

        pkb.AddTvmKey(16,
                      100506,
                      Base64url2bin("MIICVQKBgQC4a4oW39xKYw0EtrAkB2s6BYDdJwWxPXnrJ5xU380BC129oJyVPXp5lf2g6mhyo4LSyD6QoJ-NJR65-ZOprx-TnXnciOnAsxPjfhhad5wKtDQycjHtH4bSEQ__wm09LpXL5HqMOubm11n8OnQQtoEf9-K3THvy_0xkSzFow1C2zQJBANJufhSF9qTZnFlA73M3Mofzhf2AqVqe2L2uN5S9mQyhx6o8zzBm7pGiMiIea9dmOhLyS_BihHkmq4dIw40a25MCQQDgWxGhgsLCiR8R7qd7ixC669xRXyimJz5QYpWKeXhZH4fUjLIbgRRQVhKezMunkAs81yPSVwFyzla8kFTYA2AfAkEAigT7h01LzA_rL9xSyd_I17dSZkVisp7vdyxnrjD1iZqX2IPF9RTrEi7sboCaaP1hS2-G1vs1he3QNdcoTyEGegJAHAtiNDBYWFEj4j3U73FiF117iivlFMTnygxSsU8vCyPw-pGWQ3AiigrCU9mZdPIBZ5rkekrgLlnK15IKmwBsBAJAGk3PwpC-1Jsziygd7mbmUP5wv7AVK1PbF7XG8pezIZQ49UeZ5gzd0jRGREPNeuzHQl5JfgxQjyTVcOkYcaNbcgJATBRJcYoDdz5Y4Wq3Aui8cu8eMFe7iL1Q63twjW47OBjkmLKYYckGryfPHr9IISSazZHAnphq3TkCDZWRYhoMDQJASwsbzFCz7PcNqmQzPBi3foWh7PsYGgpwYINuBS4VnWhIlSaUz3y4HYIyBywVYkIfHfatfRtoT3pNrcosXH0BYg"));

        UNIT_ASSERT_STRINGS_EQUAL("1:CucECuIECAsQABrXBDCCAlMCgYBvvGVH_M2H8qxxv94yaDYUTWbRnJ1uiIYc59KIQlfFimMPhSS7x2tqUa2-hI55JiII0Xym6GNkwLhyc1xtWChpVuIdSnbvttbrt4weDMLHqTwNOF6qAsVKGKT1Yh8yf-qb-DSmicgvFc74mBQm_6gAY1iQsf33YX8578ClhKBWHQJBAKgMdsQa4-OnhQUFZSVIVqZhrdpTAXp1QcwWcq9-3YixbtJSVj9YsPzvbHDJYNRIVXhGe04aGCccNs2tsLbyeNsCQQCqNwWwtrDUcordt5_AOeIDUQgPCRawGEo_GGGZ5FRNQ4ju9VYFJ1-FlakIgoPFsN5BWJT5G_lHCiHkJV-2YAJnAkAW6n6axewmGNVBh5PkooNXx5uehvbPz-Q_A-FBVwbIEc3Vr4RhQ9sZIK3dmPnT7aM8bQRX0oePlgUDPSnEZCZ8AkAVRuC2FtYajlFbtvP4BzxAaiEB4SLWAwlH4wwzPIqJqHEd3qrApOvwsrUhEFB4thvIKxKfI38o4UQ8hKv2zABNAkAVAY7Yg1x8dPCgoKykqQrUzDW7SmAvTqg5gs5V79uxFi3aSkrH6xYfne2OGSwaiQqvCM9pw0ME44bZtbYW3k8bAkAzNH_aWmvEPIMDWJFGHtAw7cy_YexIOawv3nK1vNTFHFi0mbszbf2Tuq-AMmtwjUzO_ZJuG5wZ9TSXncFUkw62AkA5wFVodqDm6YwnFl1AVgEJiVbX8-JXSsk-jmtL5GOdsDmEfxhtoy4JIPN9p8gFlRg_hwxIjA86E-5SMgvvH_6uIJWRBhAACugECuMECAwQABrYBDCCAlQCgYEAkuzFcd5TJu7lYWYe2hQLFfUWIIj91BvQQLa_Thln4YtGCO8gG1KJqJm-YlmJOWQG0B7H_5RVhxUxV9KpmFnsDVkzUFKOsCBaYGXc12xPVioawUlAwp5qp3QQtZyx_se97YIoLzuLr46UkLcLnkIrp-Jo46QzYi_QHq45WTm8MQ0CQQDfIONiQg0UC_p1Pg4LpotgOinwlh2GfXK8FngySJ2aNNMsvLdeuUlvzUOjMt6hiT4V_bP_t3qIUC96hgcCucUTAkEAqJHtCDNjSG8BB1Eldy__Oe2JIxUjuwEbS7HO_JNA99-eyR66C0T9i302Nh8GR8SNwbA5nCzMfp_buXSTp4iVXwJAGIK5xkz5FM4oiepS2XChyl5qqVnZg3cTemM0gCAoeqdh8oBjV5DIbDg0Qfr9dmXVLVUcTqcHxupbLCQte_jOqwJAFRI9oQZsaQ3gIOokruX_5z2xJGKkd2AjaXY535JoHvvz2SPXQWifsW-mxsPgyPiRuDYHM4WZj9P7dy6SdPESrAJAG-QcbEhBooF_TqfBwXTRbAdFPhLDsM-uV4LPBkkTs0aaZZeW69cpLfmodGZb1DEnwr-2f_bvUQoF71DA4Fc4ogJAYpXo2cwGLqdY0MhMNTvXzc0Cyy-OH4gSOuxLwko7NFE1uI9jBVM3ebBe8tuHmMREbkY2mSxYhuqmzTqmmEeGmAJAep7KpsXc9gTQiVnrpO05jY3OGu-qh3PJiCYthxpr54xEcjwkcIPbNfJN7pbNCzKdNamcdysvTslV7dEn8PbIyCCWkQYQAgroBArjBAgNEAAa2AQwggJUAoGBAIUzbxOknXf_rNt17_ir8JlWvrtnCWsQd1MAnl5mgArvavDtKeBYHzi5_Ak7DHlLzuA6YE8W175FxLFKpN2hkz-l-M7ltUSd8N1BvJRhK4t6WffWfC_1wPyoAbeSN2Yb1jygtZJQ8wGoXHcJQUXiMit3eFNyylwsJFj1gzAR4JCdAkEApXaUdEFiMWSs_RgqOHY5F8xZr9lWzIr71knNV_5Jl2eZA0akvRwpx_eVQ5d4zE5MILcd_AKHJlT3UK_kZ1WbAwJBAM4VrqkieS0_1BsDW-dmCX48vMC0kGtibUkvE0R_b7KrUg2aRdeOTv5nLeYcGmHcBeRTYnV9C3xiExDTbSFUg98CQHhbqFHs8fRhWAmEToeePdT_xsSZ288PtREa292So0z5ylHpYIdrK3ZatFoJGyVVPTfjixMdreTLI_K3asWsKrwCQBnCtdUkTyWn-oNga3zswS_Hl5gWkg1sTakl4miP7fZVakGzSLrxyd_M5bzDg0w7gLyKbE6voW-MQmIabaQqkHwCQBSu0o6ILEYslZ-jBUcOxyL5izX7KtmRX3rJOar_yTLs8yBo1JejhTj-8qhy7xmJyYQW47-AUOTKnuoV_Izqs2ACQE6NrAffRAlJffgoqrI0NwKJr0bI3JqL7rtMhLsNW8TD-4o0-bq83OVogYj-Q2jSwc0LLRC1AEqjaPOwLbfQFV0CQDybyJmEv73xwD1Vq0FZMpX1UIwCOWeq0m7EML0H7bbJuGjNr-taeNbK8TSgkXOJ8WN5PUbDqAak3v8QMFEEdUQgl5EGEAEK5wQK4gQIDhAAGtcEMIICUwKBgFMcbEpl9ukVR6AO_R6sMyiU11I8b8MBSUCEC15iKsrVO8v_m47_TRRjWPYtQ9eZ7o1ocNJHaGUU7qqInFqtFaVnIceP6NmCsXhjs3MLrWPS8IRAy4Zf4FKmGOx3N9O2vemjUygZ9vUiSkULdVrecinRaT8JQ5RG4bUMY04XGIwFAkEAhqoUENwvyMLST7AofTdCAd1fcb2H72qWmQy3ir_FsAR9fUUNslMln-WyBmy9eeMzNpjzejRgLzqtob9u931-YwJBAJ3-9a22nOM3OoA8L0z9210h5JwnMUi5X_pFHoshKY7gvgDI_wz3rj2CIltemG-HYLSt3Tnw8AOfqQJKZGm1xHcCQAXO9oT-aWpMS8QM6k7WzR_AguMY48KPVzOlb-SdbtxTPiVaYZ9oPoYrumcB170KjS4_fj-5aiTqrP4cPgycDTgCQBO_3rW205xm51AHhemfu2ukPJOE5ikXK_9Io9FkJTHcF8AZH-Ge9cewREtr0w3w7BaVu6c-HgBz9SBJTI02uI8CQBDVQoIbhfkYWkn2BQ-m6EA7q-43sP3tUtMhlvFX-LYAj6-oobZKZLP8tkDNl688ZmbTHm9GjAXnVbQ37d7vr8wCQEmwF8MCrJ1cXQ6Eb9bhaWFQ-ivfftXjqC4-1SRjPfdwYPwWFGLpqv256ADkeiD2noYKCCbmqvFFJ96qtKuW_ekCQGn6iKuVXJd9n-hhSIkTxW28VPnfhDA3H8HWDH_RTpRMlLc8IytkIo2cjNuWO-uAY2UZp98WDKCzFdchG0213v0gmJEGEAMK5wQK4gQIDxAAGtcEMIICUwKBgGpCkW-NR3li8GlRvqpq2YZGSIgm_PTyDI2Zwfw69grsBmPpVFW48Vw7xoMN35zcrojEpialB_uQzlpLYOvsMl634CRIuj-n1QE3-gaZTTTE8mg-AR4mcxnTKThPnRQpbuOlYAnriwiasWiQEMbGjq_HmWioYYxFo9USlklQn4-9AkEAyZBz_sQPZW5HSxGmbspszDzYpbitf7R_5E5ZM1zYlq_5_uElqNHh8AylODpshBAi9JiMi0KTMeUy716_hpaD6wJBAIb1ERyyO1sZ-eS7pJE5kfyOyPkNWClgFrsTQ5q7c4FfG1IqS5z2HqYxG7TvN_3QqonTL1IhiIFCLPJo5BRI2PcCQEhzOLNxP1z1o9GbuclKvWpCZP9j1Z-Y7slbLbu2MnvHAmEY0vdTmsWbSmMMxi4OgXm-o09Sj8xVELPUd6lH6cACQBDeoiOWR2tjPzyXdJInMj-R2R8hqwUsAtdiaHNXbnAr42pFSXOew9TGI3ad5v-6FVE6ZepEMRAoRZ5NHIKJGx8CQBkyDn_YgeytyOliNM3ZTZmHmxS3Fa_2j_yJyyZrmxLV_z_cJLUaPD4BlKcHTZCCBF6TEZFoUmY8pl3r1_DS0H0CQFvrLWyeai0S-pgekBNdAQyiruBU5fachNxUnJRb3CzPv0iWKw9OmDm4Mp7zBQAskf02AyrLPaveQqT1w1vKE-gCQCgBq_XA-IqujM9UjF_PIa5z6JgBGrsXREeOFcDggfXPxbSxXj7tQdTCKvGR272l0b21MD7Q6njXhoeezWA_rh4gmZEGEAQS5wQK5AQIEBAAGtkEMIICVQKBgQC4a4oW39xKYw0EtrAkB2s6BYDdJwWxPXnrJ5xU380BC129oJyVPXp5lf2g6mhyo4LSyD6QoJ-NJR65-ZOprx-TnXnciOnAsxPjfhhad5wKtDQycjHtH4bSEQ__wm09LpXL5HqMOubm11n8OnQQtoEf9-K3THvy_0xkSzFow1C2zQJBANJufhSF9qTZnFlA73M3Mofzhf2AqVqe2L2uN5S9mQyhx6o8zzBm7pGiMiIea9dmOhLyS_BihHkmq4dIw40a25MCQQDgWxGhgsLCiR8R7qd7ixC669xRXyimJz5QYpWKeXhZH4fUjLIbgRRQVhKezMunkAs81yPSVwFyzla8kFTYA2AfAkEAigT7h01LzA_rL9xSyd_I17dSZkVisp7vdyxnrjD1iZqX2IPF9RTrEi7sboCaaP1hS2-G1vs1he3QNdcoTyEGegJAHAtiNDBYWFEj4j3U73FiF117iivlFMTnygxSsU8vCyPw-pGWQ3AiigrCU9mZdPIBZ5rkekrgLlnK15IKmwBsBAJAGk3PwpC-1Jsziygd7mbmUP5wv7AVK1PbF7XG8pezIZQ49UeZ5gzd0jRGREPNeuzHQl5JfgxQjyTVcOkYcaNbcgJATBRJcYoDdz5Y4Wq3Aui8cu8eMFe7iL1Q63twjW47OBjkmLKYYckGryfPHr9IISSazZHAnphq3TkCDZWRYhoMDQJASwsbzFCz7PcNqmQzPBi3foWh7PsYGgpwYINuBS4VnWhIlSaUz3y4HYIyBywVYkIfHfatfRtoT3pNrcosXH0BYiCakQY",
                                  pkb.SerializeV1());
    }
}
