#include <library/cpp/http/simple/http_client.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/tvmauth/client/facade.h>
#include <library/cpp/tvmauth/client/misc/api/threaded_updater.h>

#include <util/stream/file.h>

using namespace NTvmAuth;

class TTestClient: public NTvmApi::TThreadedUpdater {
public:
    TTestClient(const NTvmApi::TClientSettings& settings, TLoggerPtr logger)
        : TThreadedUpdater(settings, logger)
    {
        Init();
        StartWorker();
    }
};

class Logger: public ILogger {
public:
    void Log(int lvl, const TString& msg) override {
        Stream << lvl << ": " << msg << Endl;
    }

    TStringStream Stream;
};

Y_UNIT_TEST_SUITE(TvmApiClientTest) {
    TIntrusivePtr<Logger> CreateLogger() {
        return new Logger;
    }

    static ui16 GetPort() {
        TFileInput port("tvmapi.port");
        return IntFromString<ui16, 10>(port.ReadAll());
    }

    Y_UNIT_TEST(badPort) {
        auto logger = CreateLogger();
        NTvmApi::TClientSettings s;
        s.SetSelfTvmId(123);
        s.EnableServiceTicketChecking();
        s.SetTvmHostPort("localhost", 0);
        UNIT_ASSERT_EXCEPTION_CONTAINS_C(TTestClient(s, logger),
                                         TRetriableException,
                                         "can not connect to ",
                                         logger->Stream.Str());
    }

    Y_UNIT_TEST(badSrc) {
        auto logger = CreateLogger();
        NTvmApi::TClientSettings s;
        s.SetSelfTvmId(123);
        s.EnableServiceTicketChecking();
        s.EnableServiceTicketsFetchOptions("TNMsfcyRIAGpcHk4Ta5L-g", {11});
        s.SetTvmHostPort("localhost", GetPort());
        UNIT_ASSERT_EXCEPTION_CONTAINS_C(TTestClient(s, logger),
                                         TNonRetriableException,
                                         R"(Failed to start TvmClient. Do not retry: ServiceTickets: Path:/2/ticket.Code=400: {"request_id":"","status":"ERR_CLIENT","error":"NOT_FOUND","desc":"Src is not found: it never existed. Or it was created a moment ago and tvm-api needs couple minutes to get it. client_id=123"})",
                                         logger->Stream.Str());
    }

    Y_UNIT_TEST(badDst) {
        auto logger = CreateLogger();
        NTvmApi::TClientSettings s;
        s.SetSelfTvmId(39);
        s.EnableServiceTicketChecking();
        s.EnableServiceTicketsFetchOptions("TNMsfcyRIAGpcHk4Ta5L-g", {11});
        s.SetTvmHostPort("localhost", GetPort());
        UNIT_ASSERT_EXCEPTION_CONTAINS_C(TTestClient(s, logger),
                                         TNonRetriableException,
                                         "Failed to start TvmClient. Do not retry: ServiceTickets: Failed to get ServiceTicket for 11: Dst is not found: it never existed. Or it was created a moment ago and tvm-api needs couple minutes to get it. client_id=11",
                                         logger->Stream.Str());
    }

    Y_UNIT_TEST(badSecret) {
        auto logger = CreateLogger();
        NTvmApi::TClientSettings s;
        s.SetSelfTvmId(39);
        s.EnableServiceTicketChecking();
        s.EnableServiceTicketsFetchOptions("aaa", {{"dest", 27}});
        s.SetTvmHostPort("localhost", GetPort());
        UNIT_ASSERT_EXCEPTION_CONTAINS_C(TTestClient(s, logger),
                                         TNonRetriableException,
                                         R"(Failed to start TvmClient. Do not retry: ServiceTickets: Path:/2/ticket.Code=400: {"request_id":"","status":"ERR_REQUEST","error":"BROKEN_SIGN","desc":"Signature is bad: common reason is bad tvm_secret or tvm_id\/tvm_secret mismatch. https:\/\/wiki.yandex-team.ru\/passport\/tvm2\/400\/"})",
                                         logger->Stream.Str());
    }

    Y_UNIT_TEST(okOperation) {
        auto logger = CreateLogger();
        NTvmApi::TClientSettings s;
        s.SetSelfTvmId(39);
        s.EnableServiceTicketChecking();
        s.EnableServiceTicketsFetchOptions("TNMsfcyRIAGpcHk4Ta5L-g", {{"dest_27", 27}});
        s.SetTvmHostPort("localhost", GetPort());
        TTvmClient c(new TTestClient(s, logger));
        UNIT_ASSERT_EQUAL(TClientStatus::Ok, c.GetStatus().GetCode());

        auto logger2 = CreateLogger();
        NTvmApi::TClientSettings s2;
        s2.SetSelfTvmId(27);
        s2.EnableServiceTicketChecking();
        s2.EnableServiceTicketsFetchOptions("2LN58F3JHFD3dv4u6NlDbg", {{"dest_39", 39}});
        s2.SetTvmHostPort("localhost", GetPort());
        TTvmClient c2(new TTestClient(s2, logger2));
        UNIT_ASSERT_EQUAL(TClientStatus::Ok, c2.GetStatus().GetCode());

        UNIT_ASSERT(c2.CheckServiceTicket(c.GetServiceTicketFor(27)));
        UNIT_ASSERT(c2.CheckServiceTicket(c.GetServiceTicketFor("dest_27")));
        UNIT_ASSERT(c.CheckServiceTicket(c2.GetServiceTicketFor(39)));
        UNIT_ASSERT(c.CheckServiceTicket(c2.GetServiceTicketFor("dest_39")));

        UNIT_ASSERT_EXCEPTION_CONTAINS(c2.GetServiceTicketFor("ololo"),
                                       TBrokenTvmClientSettings,
                                       "Destination 'ololo' was not specified in settings. Check your settings (if you use Qloud/YP/tvmtool - check it's settings)");
        UNIT_ASSERT_EXCEPTION_CONTAINS(c2.GetServiceTicketFor(123),
                                       TBrokenTvmClientSettings,
                                       "Destination '123' was not specified in settings. Check your settings (if you use Qloud/YP/tvmtool - check it's settings)");
    }
}
