package cache

import (
	"context"
	"fmt"
	"io/fs"
	"io/ioutil"
	"os"
	"path"
	"time"

	"github.com/gofrs/flock"

	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type Manager struct {
	filePath     string
	lockFilePath string
}

func CheckDir(dir string) error {
	stat, err := os.Stat(dir)
	if err != nil {
		return err
	}
	if !stat.Mode().IsDir() {
		return fmt.Errorf("directory does not exist: %s", dir)
	}
	return nil
}

func NewManager(filepath string) (*Manager, error) {
	if !path.IsAbs(filepath) {
		return nil, fmt.Errorf("path must be absolute: %s", filepath)
	}

	filepath = path.Clean(filepath)
	lockFilePath := filepath + ".lock"
	dir := path.Dir(filepath)

	if err := CheckDir(dir); err != nil {
		return nil, err
	}

	return &Manager{
		filePath:     filepath,
		lockFilePath: lockFilePath,
	}, nil
}

func (manager *Manager) tryWithLock(f func() error) error {
	lock := flock.New(manager.lockFilePath)

	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()

	ok, err := lock.TryLockContext(ctx, 100*time.Millisecond)
	defer func() {
		err = lock.Unlock()
		if err != nil {
			logger.Log().Errorf("cannot release file lock (%s): %s", manager.lockFilePath, err)
		}
	}()

	if !ok || err != nil {
		return fmt.Errorf("unable to lock file (%s): %s", manager.lockFilePath, err)
	}

	return f()
}

func (manager *Manager) TryWrite(data []byte) error {
	return manager.tryWithLock(func() error {
		err := ioutil.WriteFile(manager.filePath, data, fs.ModePerm)
		if err != nil {
			return err
		}
		return nil
	})
}

func (manager *Manager) TryRead() ([]byte, error) {
	var data []byte
	var err error
	err = manager.tryWithLock(func() error {
		data, err = ioutil.ReadFile(manager.filePath)
		if err != nil {
			return err
		}
		return nil
	})
	if err != nil {
		return nil, err
	}
	return data, nil
}
