package krl

import (
	"sync"
	"time"

	"github.com/stripe/krl"
)

type FetchKRLFunc = func() (*krl.KRL, time.Time, string, error)

type Cache struct {
	KRL       []byte    `json:"krl"`
	Timestamp time.Time `json:"timestamp"`
	Etag      string    `json:"etag"`
}

type Holder struct {
	KRL       *krl.KRL
	Timestamp time.Time
	Etag      string
	rwLock    sync.RWMutex
}

func (holder *Holder) UpdateWithTimestamp(k *krl.KRL, ts time.Time, etag string) {
	holder.rwLock.Lock()
	defer holder.rwLock.Unlock()
	holder.KRL = k
	holder.Timestamp = ts
	holder.Etag = etag
}

func (holder *Holder) UpdateTimestampOnly(ts time.Time) {
	holder.rwLock.Lock()
	defer holder.rwLock.Unlock()
	holder.Timestamp = ts
}

func (holder *Holder) Update(k *krl.KRL, etag string) {
	holder.UpdateWithTimestamp(k, time.Now(), etag)
}

func (holder *Holder) GetKRL() (*krl.KRL, time.Time, string) {
	holder.rwLock.RLock()
	defer holder.rwLock.RUnlock()
	return holder.KRL, holder.Timestamp, holder.Etag
}

func (holder *Holder) GetEtag() string {
	holder.rwLock.RLock()
	defer holder.rwLock.RUnlock()
	return holder.Etag
}
