package tvmcert

import (
	"context"
	"fmt"
	"net/http"
	"strings"
	"time"

	"github.com/labstack/echo/v4"
	"golang.org/x/crypto/ssh"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tvmcert/internal/keys"
	"a.yandex-team.ru/passport/infra/daemons/tvmcert/internal/krl"
	"a.yandex-team.ru/passport/infra/daemons/tvmcert/internal/task"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/middlewares"
	"a.yandex-team.ru/passport/shared/golibs/juggler"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type Tvmcert struct {
	tvm             tvm.Client
	unistat         stats
	accessLog       log.Logger
	keysFetcher     keys.Fetcher
	keysFetcherTask *task.RegularTask
	krlFetcher      krl.Fetcher
	krlFetcherTask  *task.RegularTask
	checker         ssh.CertChecker
}

type Config struct {
	Keys      keys.SkottyFetcherConfig `json:"skotty_keys"`
	KRL       krl.SkottyFetcherConfig  `json:"skotty_krl"`
	AccessLog logger.Config            `json:"access_logger"`
}

type stats struct {
}

const (
	TvmReqIDCtxKey string = "tvm_request_id"
)

func TvmContextReqID(ctx context.Context) string {
	for _, k := range ctxlog.ContextFields(ctx) {
		if k.Key() == TvmReqIDCtxKey {
			return k.String()
		}
	}

	return ""
}

type Factory struct{}

func (f *Factory) NewService(config httpdaemon.ServiceConfig) (httpdaemon.Service, error) {
	var cfg Config
	if err := httpdaemon.ParseServiceConfig(config, &cfg); err != nil {
		return nil, err
	}

	keysFetcher, err := keys.NewSkottyFetcher(cfg.Keys)
	if err != nil {
		return nil, err
	}

	keysTask := task.NewRegularTask(keysFetcher, time.Minute, "update_keys")
	keysTask.Start()

	krlFetcher, err := krl.NewSkottyFetcher(cfg.KRL)
	if err != nil {
		return nil, err
	}

	krlTask := task.NewRegularTask(krlFetcher, time.Minute, "update_krl")
	krlTask.Start()

	accessLog, err := logger.CreateLog(cfg.AccessLog)
	if err != nil {
		return nil, err
	}

	res := &Tvmcert{
		unistat:     stats{},
		keysFetcher: keysFetcher,
		krlFetcher:  krlFetcher,
		accessLog:   accessLog,
		checker: ssh.CertChecker{
			IsUserAuthority: func(auth ssh.PublicKey) bool {
				err := keysFetcher.CheckPublicKey(auth)
				if err != nil {
					logger.Log().Debugf("certificate not user authority: %s", err)
					return false
				}
				return true
			},
			IsRevoked: func(cert *ssh.Certificate) bool {
				err := krlFetcher.CheckCertificate(cert)
				if err != nil {
					logger.Log().Debugf("certificate revoked: %s", err)
					return true
				}
				return false
			},
		},
	}
	return res, nil
}

func (t *Tvmcert) AddHandlers(e *echo.Echo) {
	e.Pre(
		t.middlewareAccessLog(),
		t.middlewareSendReqID(),
	)

	e.GET(
		"/ping",
		t.HandlePing(),
	)

	e.GET(
		"/healthcheck",
		t.HandleHealthCheck(),
	)

	e.POST(
		"/1/verify",
		t.HandleVerifyCertificate(),
		t.middlewareAddTvmReqID(),
		t.middlewareHandleErrors(),
	)
}

func (t *Tvmcert) middlewareAccessLog() echo.MiddlewareFunc {
	omitIfEmpty := func(val string) string {
		if len(val) == 0 {
			return ""
		}
		return val
	}

	formatSign := func(sign string) string {
		return fmt.Sprintf("%s...", sign[0:len(sign)/2])
	}

	makeParams := func(r *http.Request) string {
		data := make([]string, 0, len(r.Form))
		for key, value := range r.Form {
			if key == "sign" {
				for i := range value {
					value[i] = formatSign(value[i])
				}
			}

			data = append(data, fmt.Sprintf("%s=%s", key, strings.Join(value, ",")))
		}
		return strings.Join(data, "&")
	}

	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			c.Response().After(func() {
				startTime := middlewares.ContextStartInstant(c.Request().Context())

				t.accessLog.Debugf(
					"%s\t%s\t%s\t%s\t%.1fms\t%d\t%d\t%s\t%s",
					middlewares.ContextReqID(c.Request().Context()),
					TvmContextReqID(c.Request().Context()),
					c.RealIP(),
					c.Request().URL.Path,
					float64(time.Since(startTime).Microseconds())/1000,
					c.Response().Status,
					c.Response().Size,
					omitIfEmpty(c.Request().URL.RawQuery),
					makeParams(c.Request()),
				)
			})

			return next(c)
		}
	}
}

func (t *Tvmcert) middlewareSendReqID() echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			c.Response().Header().Set(
				echo.HeaderXRequestID,
				middlewares.ContextReqID(c.Request().Context()),
			)

			return next(c)
		}
	}
}

func (t *Tvmcert) middlewareAddTvmReqID() echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			c.SetRequest(c.Request().WithContext(
				ctxlog.WithFields(c.Request().Context(), log.String("tvm_request_id", c.FormValue("request_id"))),
			))
			return next(c)
		}
	}
}

func (t *Tvmcert) HandlePing() echo.HandlerFunc {
	return func(c echo.Context) error {
		logger.Log().Debugf("Ping: service is up")
		return c.String(http.StatusOK, "")
	}
}

func (t *Tvmcert) HandleHealthCheck() echo.HandlerFunc {
	return func(c echo.Context) error {
		status := juggler.NewStatusOk()
		status.Update(t.krlFetcher.GetJugglerStatus())
		status.Update(t.keysFetcher.GetJugglerStatus())
		return c.String(http.StatusOK, status.String())
	}
}

func (t *Tvmcert) middlewareHandleErrors() echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) error {
			err := next(c)
			if err != nil {
				ctxlog.Debugf(c.Request().Context(), logger.Log(), err.Error())
				return c.JSON(http.StatusBadRequest, err)
			}
			return nil
		}
	}
}

func (t *Tvmcert) HandleVerifyCertificate() echo.HandlerFunc {
	return func(c echo.Context) error {
		request := VerifyCertificateRequest{}
		if err := c.Bind(&request); err != nil {
			return &VerificationStatus{
				Status:  InvalidParams,
				Message: err.Error(),
			}
		}

		if len(request.Certificate.Content) == 0 ||
			len(request.Sign.Content) == 0 ||
			request.Data == "" ||
			request.Username == "" {
			return &VerificationStatus{
				Status:  InvalidParams,
				Message: "required params missing",
			}
		}

		if err := t.VerifyCertificate(request); err != nil {
			return err
		}

		return c.JSON(http.StatusOK, &VerificationStatus{
			Status: Ok,
		})
	}
}
