package tvmcert

import (
	"encoding/base64"
	"fmt"

	"golang.org/x/crypto/ssh"
)

type Base64URLString struct {
	Content []byte
}

func (s *Base64URLString) UnmarshalParam(param string) error {
	var err error
	s.Content, err = base64.RawURLEncoding.DecodeString(param)
	return err
}

type VerifyCertificateRequest struct {
	Username    string          `form:"username"`
	Data        string          `form:"data"`
	Certificate Base64URLString `form:"certificate"`
	Sign        Base64URLString `form:"sign"`
}

func (t *Tvmcert) VerifyCertificate(request VerifyCertificateRequest) error {
	publicKey, err := ssh.ParsePublicKey(request.Certificate.Content)
	if err != nil {
		return &VerificationStatus{
			Status:  InvalidType,
			Message: err.Error(),
		}
	}

	cert, ok := publicKey.(*ssh.Certificate)
	if !ok {
		return &VerificationStatus{
			Status:  InvalidType,
			Message: fmt.Sprintf("not a certificate %T", publicKey),
		}
	}

	if cert.CertType != ssh.UserCert {
		return &VerificationStatus{
			Status:  InvalidCertType,
			Message: fmt.Sprintf("invalid certificate type: %d", cert.CertType),
		}
	}

	if !t.checker.IsUserAuthority(cert.SignatureKey) {
		return &VerificationStatus{
			Status:  InvalidCA,
			Message: "not user authority",
		}
	}

	if err = t.checker.CheckCert(request.Username, cert); err != nil {
		return &VerificationStatus{
			Status:  InvalidCert,
			Message: err.Error(),
		}
	}

	sshSign := ssh.Signature{
		Format: cert.Key.Type(),
		Blob:   request.Sign.Content,
	}
	if err = cert.Verify([]byte(request.Data), &sshSign); err != nil {
		return &VerificationStatus{
			Status:  InvalidSignature,
			Message: err.Error(),
		}
	}

	return nil
}
