package mockskotty

import (
	"bytes"
	"crypto/md5"
	"crypto/rand"
	"encoding/base64"
	"encoding/hex"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/klauspost/compress/zstd"
	"github.com/stretchr/testify/require"
	"github.com/stripe/krl"
	"golang.org/x/crypto/ssh"

	mockssh "a.yandex-team.ru/passport/infra/daemons/tvmcert/mock/mockssh"
)

func New(t *testing.T, keys []mockssh.ECDSAKeyPair, k *krl.KRL) (*httptest.Server, string) {
	buffer := bytes.Buffer{}
	var signers []ssh.Signer

	for i, keyPair := range keys {
		signers = append(signers, keyPair.Signer)

		key := keyPair.PublicKey
		buffer.WriteString(key.Type())
		buffer.WriteByte(' ')
		buffer.WriteString(base64.StdEncoding.EncodeToString(key.Marshal()))
		if i != len(keys)-1 {
			buffer.WriteByte('\n')
		}
	}

	rawKRL, err := k.Marshal(rand.Reader, signers...)
	require.NoError(t, err)
	encoder, err := zstd.NewWriter(
		bytes.NewBuffer(nil),
		zstd.WithEncoderConcurrency(1))
	require.NoError(t, err)

	encodedKRL := encoder.EncodeAll(rawKRL, nil)
	checkSum := md5.Sum(encodedKRL)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		switch r.RequestURI {
		case "/api/v1/ca/pub/ssh":
			_, _ = w.Write(buffer.Bytes())
		case "/api/v1/ca/krl/all.zst":
			w.Header().Set("Etag", hex.EncodeToString(checkSum[:]))
			_, _ = w.Write(encodedKRL)
		}
	}))

	return server, server.URL
}
