package mockssh

import (
	"crypto/ecdsa"
	"crypto/elliptic"
	"crypto/rand"
	"crypto/rsa"
	"io/ioutil"
	"strconv"
	"testing"
	"time"

	"github.com/stretchr/testify/require"
	"golang.org/x/crypto/ssh"
)

func GetPrivateKeyFromFile(t *testing.T, filename string) interface{} {
	privateKeyBuf, err := ioutil.ReadFile(filename)
	require.NoError(t, err)
	key, err := ssh.ParseRawPrivateKey(privateKeyBuf)
	require.NoError(t, err)
	return key
}

type ECDSAKeyPair struct {
	PrivateKey *ecdsa.PrivateKey
	PublicKey  ssh.PublicKey
	Signer     ssh.Signer
}

func GenECDSAKey(t *testing.T) ECDSAKeyPair {
	ecdsaKey, err := ecdsa.GenerateKey(elliptic.P256(), rand.Reader)
	require.NoError(t, err)
	publicKey, err := ssh.NewPublicKey(ecdsaKey.Public())
	require.NoError(t, err)
	signer, err := ssh.NewSignerFromKey(ecdsaKey)
	require.NoError(t, err)
	return ECDSAKeyPair{
		PrivateKey: ecdsaKey,
		PublicKey:  publicKey,
		Signer:     signer,
	}
}

func GetECDSAKeyFromFile(t *testing.T, filename string) ECDSAKeyPair {
	key := GetPrivateKeyFromFile(t, filename)
	ecdsaKey, ok := key.(*ecdsa.PrivateKey)
	require.True(t, ok)
	publicKey, err := ssh.NewPublicKey(ecdsaKey.Public())
	require.NoError(t, err)
	signer, err := ssh.NewSignerFromKey(ecdsaKey)
	require.NoError(t, err)
	return ECDSAKeyPair{
		PrivateKey: ecdsaKey,
		PublicKey:  publicKey,
		Signer:     signer,
	}
}

type RSAKeyPair struct {
	PrivateKey *rsa.PrivateKey
	PublicKey  ssh.PublicKey
	Signer     ssh.Signer
}

func GenRSAKey(t *testing.T) RSAKeyPair {
	rsaKey, err := rsa.GenerateKey(rand.Reader, 4096)
	require.NoError(t, err)
	publicKey, err := ssh.NewPublicKey(rsaKey.Public())
	require.NoError(t, err)
	signer, err := ssh.NewSignerFromKey(rsaKey)
	require.NoError(t, err)
	return RSAKeyPair{
		PrivateKey: rsaKey,
		PublicKey:  publicKey,
		Signer:     signer,
	}
}

func GetRSAKeyFromFile(t *testing.T, filename string) RSAKeyPair {
	key := GetPrivateKeyFromFile(t, filename)
	rsaKey, ok := key.(*rsa.PrivateKey)
	require.True(t, ok)
	publicKey, err := ssh.NewPublicKey(rsaKey.Public())
	require.NoError(t, err)
	signer, err := ssh.NewSignerFromKey(rsaKey)
	require.NoError(t, err)
	return RSAKeyPair{
		PrivateKey: rsaKey,
		PublicKey:  publicKey,
		Signer:     signer,
	}
}

func GenerateKeys(t *testing.T, n int) []ECDSAKeyPair {
	out := make([]ECDSAKeyPair, 0, n)
	for i := 0; i < n; i++ {
		out = append(out, GenECDSAKey(t))
	}
	return out
}

func MakeECDSACert(t *testing.T, key ssh.Signer, ca ssh.Signer, id uint64, principals []string) (*ssh.Certificate, ssh.Signer) {
	now := time.Now()
	cert := &ssh.Certificate{
		Key:             key.PublicKey(),
		Serial:          id,
		CertType:        ssh.UserCert,
		KeyId:           strconv.FormatUint(id, 10),
		ValidPrincipals: principals,
		ValidAfter:      uint64(now.Add(-time.Hour).Unix()),
		ValidBefore:     uint64(now.Add(time.Hour).Unix()),
		Permissions: ssh.Permissions{
			Extensions: map[string]string{
				"permit-pty":              "",
				"permit-user-rc":          "",
				"permit-X11-forwarding":   "",
				"permit-agent-forwarding": "",
				"permit-port-forwarding":  "",
			},
		},
	}

	err := cert.SignCert(rand.Reader, ca)
	require.NoError(t, err)

	signer, err := ssh.NewCertSigner(cert, key)
	require.NoError(t, err)

	return cert, signer
}
