package mocktvmcert

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"os"
	"os/exec"
	"testing"
	"time"

	"github.com/go-resty/resty/v2"
	"github.com/stretchr/testify/require"

	"a.yandex-team.ru/library/go/test/portmanager"
	"a.yandex-team.ru/library/go/test/yatest"
	"a.yandex-team.ru/passport/infra/daemons/tvmcert/internal/keys"
	"a.yandex-team.ru/passport/infra/daemons/tvmcert/internal/krl"
	"a.yandex-team.ru/passport/infra/daemons/tvmcert/internal/tvmcert"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/passport/shared/golibs/utils"
)

type HTTPConfig struct {
	Address string `json:"listen_address"`
	Port    int    `json:"port"`
}

type TestConfig struct {
	HTTPCommon  HTTPConfig     `json:"http_common"`
	HTTPUnistat HTTPConfig     `json:"http_unistat"`
	Logger      logger.Config  `json:"logger"`
	Service     tvmcert.Config `json:"service"`
}

func MakeConfig(t *testing.T, skottyURL string) TestConfig {
	pm := portmanager.New(t)
	return TestConfig{
		HTTPCommon: HTTPConfig{
			Address: "localhost",
			Port:    pm.GetPort(),
		},
		HTTPUnistat: HTTPConfig{
			Address: "localhost",
			Port:    pm.GetPort(),
		},
		Logger: logger.Config{
			FilePath: yatest.OutputPath("common.log"),
		},
		Service: tvmcert.Config{
			AccessLog: logger.Config{
				FilePath: yatest.OutputPath("access.log"),
			},
			Keys: keys.SkottyFetcherConfig{
				SkottyURL:        skottyURL,
				CacheDir:         yatest.OutputPath(""),
				KeysUpdatePeriod: &utils.Duration{Duration: time.Minute},
			},
			KRL: krl.SkottyFetcherConfig{
				SkottyURL:    skottyURL,
				CacheDir:     yatest.OutputPath(""),
				UpdatePeriod: &utils.Duration{Duration: time.Minute},
			},
		},
	}
}

func New(t *testing.T, config TestConfig) (*exec.Cmd, string) {
	data, err := json.Marshal(config)
	require.NoError(t, err)
	require.NoError(t, ioutil.WriteFile(yatest.OutputPath("tvmcert.conf"), data, os.ModePerm))
	daemon := exec.Command(
		yatest.BuildPath("passport/infra/daemons/tvmcert/cmd/tvmcert"),
		"-c", yatest.OutputPath("tvmcert.conf"))
	daemon.Stdout = os.Stderr
	daemon.Stderr = os.Stderr

	err = daemon.Start()
	require.NoError(t, err)

	client := resty.New().SetBaseURL(fmt.Sprintf("http://%s:%d", config.HTTPCommon.Address, config.HTTPCommon.Port))

	for i := 0; i < 50; i++ {
		response, err := client.R().Get("/ping")
		t.Logf("%s: %s", response, err)
		if err == nil && response.IsSuccess() {
			t.Logf("Tvmcert successfully started on %s", client.BaseURL)
			return daemon, client.BaseURL
		}
		time.Sleep(100 * time.Millisecond)
	}
	t.FailNow()
	return nil, ""
}
