package cli

import (
	"fmt"
	"log"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmsrv"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmtypes"
)

const (
	wikilink = "https://wiki.yandex-team.ru/passport/tvm2"
)

var addOpts struct {
	Src        string
	Dst        []string
	Secret     string
	ConfigPath string
}

var addCmd = &cobra.Command{
	Use:   "add",
	Short: "Add tvm-client configuration",
	Long:  fmt.Sprintf("Check documentation at %s", wikilink),
	PreRunE: func(cmd *cobra.Command, args []string) error {
		if cfgfile, err := initConfigFile(addOpts.ConfigPath); err == nil {
			addOpts.ConfigPath = cfgfile
		} else {
			return err
		}

		if addOpts.Secret == "" {
			return fmt.Errorf("secret must not be empty")
		}

		if addOpts.Src == "" {
			return fmt.Errorf("src must not be empty")
		}

		if len(addOpts.Dst) == 0 {
			return fmt.Errorf("dst must not be empty")
		}

		return nil
	},
	RunE: func(cmd *cobra.Command, args []string) error {
		return processAdd()
	},
}

func init() {
	flags := addCmd.Flags()
	flags.StringVarP(&addOpts.Secret, "secret", "s", "", "TVM Application Secret")
	flags.StringVar(&addOpts.Src, "src", "", "TVM_ID and Alias in id:alias format")
	flags.StringSliceVar(&addOpts.Dst, "dst", []string{}, "TVM Destination Apps id and aliases, each must be in id:alias format")
	flags.StringVarP(&addOpts.ConfigPath, "config", "c", "", "Configuration file")

	RootCmd.AddCommand(addCmd)
}

func processAdd() error {
	log.Println("Using configuration file: ", addOpts.ConfigPath)
	tvmcfg, err := readConfiguration(addOpts.ConfigPath)
	if err != nil {
		return fmt.Errorf("failed to read config: %s", err)
	}

	srcid, srcalias, err := parseSrcDst(addOpts.Src)
	if err != nil {
		return fmt.Errorf("failed to parse src: %s", err)
	}

	removeOldConfiguration(tvmcfg, srcid, srcalias)

	client := tvmsrv.ClientView{}
	client.Secret = addOpts.Secret
	client.SelfTvmID = srcid
	client.Dsts = make(map[string]tvmtypes.Dst, len(addOpts.Dst))

	for i := range addOpts.Dst {
		dstid, dstalias, err := parseSrcDst(addOpts.Dst[i])
		if err != nil {
			return err
		}

		client.Dsts[dstalias] = tvmtypes.Dst{ID: dstid}
	}

	tvmcfg.Clients[srcalias] = client

	if err := saveConfiguration(tvmcfg, addOpts.ConfigPath); err != nil {
		return fmt.Errorf("failed to save new config: %s", err)
	}

	log.Println("Configuration has been updated")
	return nil
}
