package cli

import (
	"crypto/rand"
	"encoding/hex"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"path"
	"regexp"
	"strconv"
	"strings"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmsrv"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

const (
	authTokenLen      = 32
	defaultConfigName = ".tvm.json"
	fileMod           = 0640
)

func parseSrcDst(src string) (tvm.ClientID, string, error) {
	s := strings.Split(src, ":")
	if len(s) != 2 {
		return 0, "", fmt.Errorf("incorrect format")
	}

	tvmid, err := strconv.Atoi(s[0])
	if err != nil {
		return 0, "", err
	}

	return tvm.ClientID(tvmid), s[1], nil
}

func findConfigurationFile() (string, error) {
	wrkdir, err := os.Getwd()
	if err == nil {
		configPath := path.Join(wrkdir, defaultConfigName)
		if _, serr := os.Stat(configPath); serr == nil {
			return configPath, nil
		}
	}

	wrkdir, err = os.UserHomeDir()
	if err == nil {
		configPath := path.Join(wrkdir, defaultConfigName)
		if _, serr := os.Stat(configPath); serr == nil {
			return configPath, nil
		}
	}

	return "", errors.New("configuration file wasn't found either in the workdir or homedir")
}

func initConfigFile(config string) (string, error) {
	if config != "" {
		return config, nil
	}

	config, err := findConfigurationFile()
	if err == nil {
		return config, nil
	}

	workdir, err := os.Getwd()
	if err == nil {
		return path.Join(workdir, defaultConfigName), nil
	}

	return "", err
}

func readConfiguration(filename string) (*tvmsrv.ConfigView, error) {
	if _, err := os.Stat(filename); os.IsNotExist(err) {
		log.Println("Configuration file not found, a new one will be created")
		return &tvmsrv.ConfigView{
			Clients: make(map[string]tvmsrv.ClientView),
		}, nil
	}

	data, err := ioutil.ReadFile(filename)
	if err != nil {
		return nil, err
	}

	var tvmcfg tvmsrv.ConfigView
	if err := json.Unmarshal(data, &tvmcfg); err != nil {
		return nil, err
	}

	return &tvmcfg, nil
}

func saveConfiguration(tvmcfg *tvmsrv.ConfigView, filename string) error {
	data, err := json.MarshalIndent(tvmcfg, "", "    ")
	if err != nil {
		return err
	}
	return ioutil.WriteFile(filename, data, os.FileMode(fileMod))
}

func removeOldConfiguration(config *tvmsrv.ConfigView, id tvm.ClientID, alias string) {
	if _, ok := config.Clients[alias]; ok {
		log.Printf("App with alias %s already exists. It will be replaced! \n", alias)
		delete(config.Clients, alias)
	}

	toRemove := make([]string, 1)
	for k, v := range config.Clients {
		if v.SelfTvmID == id {
			log.Printf("App with tvmid %d already exists. It will be replaced! \n", id)
			toRemove = append(toRemove, k)
		}
	}

	for i := range toRemove {
		delete(config.Clients, toRemove[i])
	}
}

func prepareOpts(opts *Opts) error {
	if opts.Config == nil {
		return fmt.Errorf("wrong configuration")
	}

	opts.provideInterface()

	if err := opts.providePort(); err != nil {
		return err
	}
	logger.Log().Infof("Using port number %d", opts.Port)

	if err := opts.provideCacheDir(); err != nil {
		return err
	}

	if err := opts.provideAuthToken(); err != nil {
		return err
	}
	if err := checkAuthToken(opts.AuthToken); err != nil {
		return err
	}

	return getSecretsInConfiguration(opts.Config)
}

func getSecretsInConfiguration(config *tvmsrv.ConfigView) error {
	for clientName, client := range config.Clients {
		var secret string
		var err error

		if strings.HasPrefix(client.Secret, "env:") {
			secret, err = getSecretFromEnv(client.Secret, clientName, client.SelfTvmID)
		} else if strings.HasPrefix(client.Secret, "json:") {
			secret, err = getSecretFromJSON(client.Secret, clientName, client.SelfTvmID)
		} else {
			continue
		}

		if err != nil {
			return err
		}
		client.Secret = secret
		config.Clients[clientName] = client
	}

	return nil
}

func getSecretFromEnv(val string, name string, id tvm.ClientID) (string, error) {
	envName := val[4:]
	newSecret := os.Getenv(envName)
	if len(newSecret) == 0 {
		return "", fmt.Errorf("failed to get secret for client %d (%s): env '%s' is empty or absent",
			id,
			name,
			envName,
		)
	}

	return newSecret, nil
}

func getSecretFromJSON(val string, name string, id tvm.ClientID) (string, error) {
	rgxRes := regexp.MustCompile(`^json:(.*)\[(.*)\]$`).FindAllSubmatchIndex([]byte(val), -1)
	if len(rgxRes) != 1 || len(rgxRes[0]) != 6 {
		return "", fmt.Errorf("failed to parse structured secret value for client %d (%s): %s", id, name, val)
	}
	filename := val[rgxRes[0][2]:rgxRes[0][3]]
	key := val[rgxRes[0][4]:rgxRes[0][5]]

	data, err := ioutil.ReadFile(filename)
	if err != nil {
		return "", err
	}

	var kv map[string]json.RawMessage
	if err = json.Unmarshal(data, &kv); err != nil {
		return "", fmt.Errorf("failed to load secret from file for client %d (%s): json map is expected in '%s'",
			id,
			name,
			filename)
	}

	secretValue, ok := kv[key]
	if !ok {
		return "", fmt.Errorf("failed to load secret from file for client %d (%s): key '%s' was not found in '%s'",
			id,
			name,
			key,
			filename)
	}

	var secret string
	if err := json.Unmarshal(secretValue, &secret); err != nil {
		return "", fmt.Errorf("failed to load secret from file for client %d (%s): key '%s' is not string in '%s'",
			id,
			name,
			key,
			filename)
	}

	if len(secret) == 0 {
		return "", fmt.Errorf("failed to load secret from file for client %d (%s): key '%s' is empty in '%s'",
			id,
			name,
			key,
			filename)
	}

	return secret, nil
}

func checkAuthToken(authtoken string) error {
	if len(authtoken) != authTokenLen {
		return fmt.Errorf(
			"authtoken has unexpected length: %d, it must be exactly %d symbols long",
			len(authtoken),
			authTokenLen,
		)
	}

	return nil
}

func generateAuthToken() (string, error) {
	random := make([]byte, authTokenLen/2)
	if _, err := rand.Read(random); err != nil {
		return "", err
	}
	return hex.EncodeToString(random), nil
}
