package cli

import (
	"fmt"
	"os"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmsrv"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmversion"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

const (
	name                = "tvmtool"
	envAuthToken        = "TVMTOOL_LOCAL_AUTHTOKEN"
	legacyEnvAuthToken  = "QLOUD_TVM_TOKEN"
	legacyEnvAuthToken2 = "TVMTOOl_LOCAL_AUTHTOKEN"
	defaultInterface    = "localhost"
)

type Opts struct {
	CfgFile           string
	Port              uint16
	Config            *tvmsrv.ConfigView
	AuthToken         string
	Verbose           bool
	Version           bool
	UseCache          bool
	CacheDir          string
	BindAllInterfaces bool
	UnittestMode      bool
	UnittestRolesDir  string
	Interface         string
}

func (o *Opts) providePort() error {
	if o.Port > 0 {
		return nil
	}
	if o.Config.Port != nil {
		o.Port = *o.Config.Port
		return nil
	}

	return fmt.Errorf("you need to specify the port to run the server using --port flag or subcommand setport")
}

func (o *Opts) provideInterface() {
	if o.Interface == "*" || (o.Interface == defaultInterface && o.BindAllInterfaces) {
		o.Interface = ""
		logger.Log().Warnf("DANGER: Tvm tool will bind to [::] instead of localhost only")
		return
	}

	if o.Interface != "127.0.0.1" && o.Interface != "::1" {
		logger.Log().Warnf("DANGER: Tvm tool will bind to [%s] instead of localhost only", o.Interface)
	}
}

func (o *Opts) provideCacheDir() error {
	if o.CacheDir != "" {
		return nil
	}

	if o.UseCache {
		var err error
		o.CacheDir, err = os.Getwd()
		if err != nil {
			return fmt.Errorf("failed to get working directory: %s", err)
		}

		return nil
	}

	return nil
}

func (o *Opts) provideAuthToken() error {
	if o.AuthToken != "" {
		return nil
	}
	logger.Log().Warnf("Auth token isn't set by user, it will be generated automatically")

	var err error
	o.AuthToken, err = generateAuthToken()
	if err != nil {
		return fmt.Errorf("failed to generate AUTH_TOKEN: %v", err)
	}
	logger.Log().Warnf("Server access token: %s", o.AuthToken)

	return nil
}

var opts Opts

func init() {
	flags := RootCmd.Flags()
	flags.StringVarP(&opts.CfgFile, "config", "c", "", "configuration file")
	flags.Uint16VarP(&opts.Port, "port", "p", 0, "port to run the service")
	flags.StringVarP(&opts.AuthToken, "auth", "a", "", fmt.Sprintf("DANGEROUS: do not use in production - use enviroment '%s'. auth token for local authentication. Must be at least 32 symbols", envAuthToken))
	flags.BoolVarP(&opts.Verbose, "verbose", "v", false, "verbose mode")
	flags.BoolVarP(&opts.Version, "version", "V", false, "print version")
	flags.BoolVarP(&opts.UseCache, "cache", "e", false, "cache keys and tickets for faster restart (in working directory)")
	flags.StringVarP(&opts.CacheDir, "cache-dir", "d", "", "cache keys and tickets for faster restart")
	flags.StringVarP(&opts.Interface, "dangerous-interface", "", defaultInterface, "DANGEROUS: bind http to this interface. '*' - bind for all")
	flags.BoolVar(&opts.BindAllInterfaces, "dangerous-bind-all-interfaces",
		false, "DANGEROUS,DEPRECATED: bind all interfaces instead of localhost only. Ignored if there is '--dangerous-interface'")
	flags.BoolVarP(&opts.UnittestMode, "unittest", "u", false, "use keys from 'tvmknife unittest' to check and generate tickets IN-MEMORY, ignore tvm-api.yandex.net")
	flags.StringVarP(&opts.UnittestRolesDir, "unittest-roles-dir", "R", "", "read roles from dir for --unittest mode")

	RootCmd.Example = `  tvmtool [command] --help
  tvmtool add --secret <YOURSECRET> --src <YOURAPPID>:<YOURALIAS> --dst <TARGETAPPID>:<TARGETALIAS> --dst <TARGETAPPID>:<TARGETALIAS> -c /etc/tvmtool/tvmtool.conf
  tvmtool add_only_for_check --tvmid <YOURAPPID>:<YOURALIAS> -c /etc/tvmtool/tvmtool.conf
  tvmtool bbenv -e 1 -c /etc/tvmtool/tvmtool.conf
  tvmtool setport -p 18080 -c /etc/tvmtool/tvmtool.conf
  tvmtool --port <portnum>`
}

func prepareAndCheck() error {
	logger.Log().Infof("tvmtool version: %s", tvmversion.GetVersion())

	if opts.Version {
		os.Exit(0)
	}

	configPath, err := initConfigFile(opts.CfgFile)
	if err != nil {
		return err
	}

	if _, err := os.Stat(configPath); os.IsNotExist(err) {
		return fmt.Errorf("you need to create configuration file manually or using 'add' command")
	}

	logger.Log().Infof("Using configuration file: %s", configPath)

	tvmcfg, err := readConfiguration(configPath)
	if err != nil {
		return err
	}
	opts.Config = tvmcfg

	if len(opts.AuthToken) != 0 {
		logger.Log().Warnf("DANGER: do not use '--auth'/'-a' in production because of security issue: auth token will be shown in `ps aux`")
	} else {
		opts.AuthToken = os.Getenv(legacyEnvAuthToken2)
		if opts.AuthToken != "" {
			logger.Log().Debugf("Got authtoken from %s", legacyEnvAuthToken2)
		}
	}
	if len(opts.AuthToken) == 0 {
		opts.AuthToken = os.Getenv(envAuthToken)
		if opts.AuthToken != "" {
			logger.Log().Debugf("Got authtoken from %s", envAuthToken)
		}
	}
	if len(opts.AuthToken) == 0 {
		opts.AuthToken = os.Getenv(legacyEnvAuthToken)
		if opts.AuthToken != "" {
			logger.Log().Debugf("Got authtoken from %s", legacyEnvAuthToken)
		}
	}

	return nil
}

var RootCmd = &cobra.Command{
	Use:          name,
	SilenceUsage: true,
	PreRunE: func(cmd *cobra.Command, args []string) error {
		return prepareAndCheck()
	},
	RunE: func(cmd *cobra.Command, args []string) error {
		if err := prepareOpts(&opts); err != nil {
			return err
		}

		return tvmsrv.StartService(
			opts.Config,
			opts.Port,
			opts.AuthToken,
			opts.Verbose,
			opts.CacheDir,
			opts.Interface,
			opts.UnittestMode,
			opts.UnittestRolesDir,
		)
	},
}

func Execute() {
	maxprocs.AdjustAuto()

	if err := RootCmd.Execute(); err != nil {
		_ = RootCmd.Usage()
		os.Exit(1)
	}
}
