package handlers

import (
	"fmt"
	"net/http"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmcontext"
)

const (
	SrvTicketHeader = "X-Ya-Service-Ticket"
)

type ServiceTicketCheckResponse struct {
	Src           tvm.ClientID `json:"src"`
	Dst           tvm.ClientID `json:"dst"`
	Scopes        []string     `json:"scopes"`
	DebugString   string       `json:"debug_string"`
	LoggingString string       `json:"logging_string"`
	IssuerUID     *uint64      `json:"issuer_uid"`
}

func makeSrvTicketCheckResponse(tkt *tvmcontext.CheckedServiceTicket) *ServiceTicketCheckResponse {
	return &ServiceTicketCheckResponse{
		Src:           tkt.SrcID,
		Dst:           tkt.DstID,
		DebugString:   tkt.DbgInfo,
		LoggingString: tkt.LogInfo,
		IssuerUID:     tkt.IssuerUID,
	}
}

type checkSrv interface {
	GetServiceContext() (*tvmcontext.ServiceContext, error)
}

func getParsedTVMServiceTicket(c checkSrv, ctx echo.Context, expectedDst tvm.ClientID) (*tvmcontext.CheckedServiceTicket, error) {
	ticket := ctx.Request().Header.Get(SrvTicketHeader)
	if ticket == "" {
		return nil, &errs.InvalidParam{
			Message: fmt.Sprintf("ticket was not found in %s header", SrvTicketHeader),
		}
	}

	srvctx, err := c.GetServiceContext()
	if err != nil {
		return nil, &errs.Temporary{Message: err.Error()}
	} else if srvctx == nil {
		return nil, &errs.Temporary{Message: "internal error: missing srvctx"}
	}

	checked, err := srvctx.CheckTicket(ticket, expectedDst)
	if err != nil {
		return nil, err
	}

	return checked, nil
}

func (t *TicketHandler) GetCheckSrvTicketHandler(c checkSrv) echo.HandlerFunc {
	return func(ctx echo.Context) error {
		dst, err := t.getSelfClient(ctx.Request(), "dst")
		if err != nil {
			return &errs.InvalidParam{Message: err.Error()}
		}

		checked, err := getParsedTVMServiceTicket(c, ctx, dst.SelfTvmID)
		if err != nil {
			return err
		}

		response := makeSrvTicketCheckResponse(checked)

		return ctx.JSON(http.StatusOK, response)
	}
}
