package handlers

import (
	"fmt"
	"net/http"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmcontext"
)

const (
	UsrTicketHeader = "X-Ya-User-Ticket"
)

var overrideRules = map[tvm.BlackboxEnv]map[tvm.BlackboxEnv]interface{}{
	tvm.BlackboxProd: {
		tvm.BlackboxProdYateam: nil,
		tvm.BlackboxProd:       nil,
	},
	tvm.BlackboxProdYateam: {
		tvm.BlackboxProd:       nil,
		tvm.BlackboxProdYateam: nil,
	},
	tvm.BlackboxTest: {
		tvm.BlackboxProd:       nil,
		tvm.BlackboxProdYateam: nil,
		tvm.BlackboxTest:       nil,
		tvm.BlackboxTestYateam: nil,
		tvm.BlackboxStress:     nil,
	},
	tvm.BlackboxTestYateam: {
		tvm.BlackboxTest:       nil,
		tvm.BlackboxTestYateam: nil,
	},
	tvm.BlackboxStress: {
		tvm.BlackboxStress: nil,
	},
}

type UserTicketResponse struct {
	DefaultUID    tvm.UID   `json:"default_uid"`
	Uids          []tvm.UID `json:"uids"`
	Scopes        []string  `json:"scopes"`
	DebugString   string    `json:"debug_string"`
	LoggingString string    `json:"logging_string"`
}

func makeUsrTicketCheckResponse(tkt *tvmcontext.CheckedUserTicket) *UserTicketResponse {
	return &UserTicketResponse{
		DefaultUID:    tkt.DefaultUID,
		Uids:          tkt.Uids,
		Scopes:        tkt.Scopes,
		DebugString:   tkt.DebugString,
		LoggingString: tkt.LoggingString,
	}
}

func getEnv(defaultEnv tvm.BlackboxEnv, overriddenEnv string) (tvm.BlackboxEnv, error) {
	if overriddenEnv == "" {
		return defaultEnv, nil
	}

	overriddenEnvEnum, err := tvm.BlackboxEnvFromString(overriddenEnv)
	if err != nil {
		return 0, fmt.Errorf("overridden env failed to be parsed: %s", err)
	}

	rules := overrideRules[defaultEnv]
	_, ok := rules[overriddenEnvEnum]
	if ok {
		return overriddenEnvEnum, nil
	}

	return defaultEnv, fmt.Errorf("can't switch env from %s to the requested %s", defaultEnv, overriddenEnv)
}

type checkUsr interface {
	GetUserContext() (*tvmcontext.UserContext, error)
}

func GetCheckUsrHandler(c checkUsr) echo.HandlerFunc {
	return func(ctx echo.Context) error {
		ticket := ctx.Request().Header.Get(UsrTicketHeader)
		if ticket == "" {
			return &errs.InvalidParam{
				Message: fmt.Sprintf("ticket was not found in %s header", UsrTicketHeader),
			}
		}

		usrctx, err := c.GetUserContext()
		if err != nil { // TODO drop err
			return &errs.Temporary{Message: err.Error()}
		} else if usrctx == nil {
			return &errs.Temporary{Message: "internal error: missing usrctx"}
		}

		env, err := getEnv(usrctx.GetDefaultEnv(), ctx.Request().URL.Query().Get("override_env"))
		if err != nil {
			return &errs.InvalidParam{Message: err.Error()}
		}

		checked, err := usrctx.CheckTicketOverriddenEnv(ticket, env)
		if err != nil {
			return err
		}

		response := makeUsrTicketCheckResponse(checked)

		return ctx.JSON(http.StatusOK, response)
	}
}
