package handlers

import (
	"fmt"
	"net/http"
	"strconv"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmtypes"
)

const (
	headerETag        = "ETag"
	headerIfNoneMatch = "If-None-Match"
)

type rolesHolder interface {
	GetRoles(slug string) (*tvm.Roles, error)
}

func GetRolesHandlerV2(cfg *tvmtypes.OptimizedConfig, cache rolesHolder) echo.HandlerFunc {
	return func(ctx echo.Context) error {
		selfAlias, currentRevision, err := getRolesHandlerV2Args(ctx.Request())
		if err != nil {
			return &errs.InvalidParam{Message: err.Error()}
		}

		roles, err := getRolesForClient(selfAlias, cfg, cache)
		if err != nil {
			return &errs.InvalidParam{
				Message: fmt.Sprintf("failed to get roles for self alias '%s': %s", selfAlias, err),
			}
		}

		if currentRevision != "" && currentRevision == roles.GetMeta().Revision {
			return ctx.NoContent(http.StatusNotModified)
		}

		ctx.Response().Header().Set(headerETag, fmt.Sprintf(`"%s"`, roles.GetMeta().Revision))
		return ctx.JSONBlob(http.StatusOK, roles.GetRaw())
	}
}

func getRolesHandlerV2Args(r *http.Request) (string, string, error) {
	selfAlias, err := getRequiredStringParam(r.URL.Query(), "self")
	if err != nil {
		return "", "", err
	}
	currentRevision, err := getCurrentRevision(r)
	if err != nil {
		return "", "", err
	}

	return selfAlias, currentRevision, nil
}

func getCurrentRevision(r *http.Request) (string, error) {
	header := r.Header.Get(headerIfNoneMatch)
	if header == "" {
		return "", nil
	}

	unquoted, err := strconv.Unquote(header)
	if err != nil {
		return "", xerrors.Errorf("failed to unquote header %s body: '%s'", headerIfNoneMatch, header)
	}

	return unquoted, nil
}

func getRolesForClient(selfAlias string, cfg *tvmtypes.OptimizedConfig, cache rolesHolder) (*tvm.Roles, error) {
	client := cfg.FindClientByAlias(selfAlias)
	if client == nil {
		return nil, xerrors.Errorf("couldn't find client in config by alias: '%s'", selfAlias)
	}

	if client.IdmSlug == "" {
		return nil, xerrors.Errorf("IDM slug is not configured for self alias '%s'", selfAlias)
	}

	roles, err := cache.GetRoles(client.IdmSlug)
	if err != nil {
		return nil, xerrors.Errorf("failed to get roles for self alias '%s': %s", selfAlias, err)
	}

	return roles, nil
}
