package handlers

import (
	"fmt"
	"runtime"

	"github.com/labstack/echo/v4"
	"github.com/shirou/gopsutil/v3/process"

	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/metrics/solomon"
	"a.yandex-team.ru/library/go/yandex/solomon/reporters/puller/httppuller"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmtypes"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmversion"
)

const (
	VersionSensorName = "version"
	VersionTagName    = "version"

	GomaxprocsSensorName     = "process.goMaxProcs"
	NumThreadsSensorName     = "process.numThreads"
	VirtualMemorySensorName  = "process.memVmsBytes"
	ResidentMemorySensorName = "process.memRssBytes"
)

type solomonMetricsHandler struct {
	registry *solomon.Registry
	selfProc *process.Process

	// Metrics
	version        metrics.Gauge
	gomaxprocs     metrics.Gauge
	numThreads     metrics.Gauge
	virtualMemory  metrics.Gauge
	residentMemory metrics.Gauge
}

func NewSolomonMetricsHandler(registry *solomon.Registry, selfProc *process.Process) echo.HandlerFunc {
	h := &solomonMetricsHandler{
		registry: registry,
		selfProc: selfProc,
	}

	h.version = registry.WithTags(map[string]string{VersionTagName: tvmversion.GetVersion()}).Gauge(VersionSensorName)
	h.version.Set(1)

	h.initSystemMetrics()

	handler := httppuller.NewHandler(h.registry)

	return func(ctx echo.Context) error {
		h.collectSystemMetrics()
		handler.ServeHTTP(ctx.Response(), ctx.Request())
		return nil
	}
}

func (h *solomonMetricsHandler) initSystemMetrics() {
	h.gomaxprocs = h.registry.Gauge(GomaxprocsSensorName)

	if h.selfProc != nil {
		h.numThreads = h.registry.Gauge(NumThreadsSensorName)
		h.virtualMemory = h.registry.Gauge(VirtualMemorySensorName)
		h.residentMemory = h.registry.Gauge(ResidentMemorySensorName)
	}
}

func (h solomonMetricsHandler) collectSystemMetrics() {
	// func GOMAXPROCS(n int) int. If n < 1, it does not change the current setting.
	h.gomaxprocs.Set(float64(runtime.GOMAXPROCS(0)))

	if h.selfProc != nil {
		numThreads, err := h.selfProc.NumThreads()
		if err == nil {
			h.numThreads.Set(float64(numThreads))
		} else {
			h.numThreads.Set(0)
		}

		memoryInfo, err := h.selfProc.MemoryInfo()
		if err == nil {
			h.virtualMemory.Set(float64(memoryInfo.VMS))
			h.residentMemory.Set(float64(memoryInfo.RSS))
		} else {
			h.virtualMemory.Set(0)
			h.residentMemory.Set(0)
		}
	}
}

func NewSolomonTVMAuthMiddleware(solomonConfig *tvmtypes.SolomonConfig, checkSrv checkSrv) echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(ctx echo.Context) error {
			if solomonConfig == nil {
				return next(ctx)
			}

			checked, err := getParsedTVMServiceTicket(checkSrv, ctx, solomonConfig.StageTVMID)
			if err != nil {
				return err
			}

			if checked.SrcID != solomonConfig.SolomonTVMID {
				return &errs.Forbidden{
					Message:       fmt.Sprintf("Wrong ticket src, expected %d, got %d", solomonConfig.SolomonTVMID, checked.DstID),
					DebugString:   checked.DbgInfo,
					LoggingString: checked.LogInfo,
				}
			}

			return next(ctx)
		}
	}
}
