package handlers

import (
	"errors"
	"fmt"
	"net/http"
	"strconv"
	"strings"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmtypes"
)

const (
	separator = ","
)

type TicketStruct struct {
	Ticket tvmtypes.Ticket `json:"ticket,omitempty"`
	ID     tvm.ClientID    `json:"tvm_id"`
	Error  string          `json:"error,omitempty"`
}

type TicketsResponse map[string]TicketStruct

type TicketErrorResponse struct {
	Error         string `json:"error"`
	DebugString   string `json:"debug_string"`
	LoggingString string `json:"logging_string"`
}

type TicketHandler struct {
	cfg        *tvmtypes.OptimizedConfig
	canOmitSrc bool
	selfClient *tvmtypes.Client
}

func NewTicketHandler(config *tvmtypes.OptimizedConfig) TicketHandler {
	t := TicketHandler{
		cfg:        config,
		canOmitSrc: config.GetNumClients() == 1,
	}

	if t.canOmitSrc {
		t.selfClient = config.GetClients()[0]
	}

	return t
}

func (t *TicketHandler) getSelfClient(r *http.Request, name string) (*tvmtypes.Client, error) {
	if t.canOmitSrc {
		return t.selfClient, nil
	}

	src := r.URL.Query().Get(name)
	if src == "" {
		return nil, fmt.Errorf("missing parameter '%s'", name)
	}

	if client := t.cfg.FindClientByAlias(src); client != nil {
		return client, nil
	}

	/* Probably not an alias */
	srcNum, err := strconv.Atoi(src)
	if err != nil {
		return nil, fmt.Errorf("'%s' parameter incorrect", name)
	}

	if client := t.cfg.FindClientByID(tvm.ClientID(srcNum)); client != nil {
		return client, nil
	}

	return nil, fmt.Errorf("couldn't find in config client by either src or alias for param = %s", src)
}

func getDsts(r *http.Request) ([]string, error) {
	dstsq := r.URL.Query().Get("dsts")
	if dstsq == "" {
		return nil, errors.New("missing parameter 'dsts'")
	}

	return strings.Split(dstsq, separator), nil
}

type getTickets interface {
	GetTicketUpdateTime() time.Time
	GetTicket(src tvm.ClientID, dst tvm.ClientID) (tvmtypes.Ticket, string, error)
}

func (t *TicketHandler) GetTicketHandler(c getTickets) echo.HandlerFunc {
	return func(ctx echo.Context) error {
		client, err := t.getSelfClient(ctx.Request(), "src")
		if err != nil {
			return &errs.InvalidParam{Message: err.Error()}
		}

		dsts, err := getDsts(ctx.Request())
		if err != nil {
			return &errs.InvalidParam{Message: err.Error()}
		}

		updateTime := c.GetTicketUpdateTime()
		response := make(TicketsResponse, len(dsts))
		for i := range dsts {
			dstID, dstAlias, err := t.cfg.FindDstForClient(client, dsts[i])
			if err != nil {
				return &errs.InvalidParam{Message: err.Error()}
			}

			ticket, er, err := c.GetTicket(client.SelfTvmID, dstID)
			if err != nil {
				return &errs.InvalidParam{Message: err.Error()}
			}

			response[dstAlias] = TicketStruct{Ticket: ticket, ID: dstID, Error: er}
		}

		ctx.Response().Header().Set(birthTimeHeader, strconv.FormatInt(updateTime.Unix(), 10))
		return ctx.JSON(http.StatusOK, response)
	}
}
