package handlers

import (
	"fmt"
	"net/http"
	"strconv"
	"strings"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/tvmtool/internal/tvmtypes"
)

func getTicketHandlerV2Args(r *http.Request) (string, []string, error) {
	query := r.URL.Query()

	selfAlias, err := getRequiredStringParam(query, "self")
	if err != nil {
		return "", nil, err
	}

	dsts, err := getRequiredStringParam(query, "dsts")
	if err != nil {
		return "", nil, err
	}

	return selfAlias, strings.Split(dsts, separator), nil
}

func GetTicketHandlerV2(cfg *tvmtypes.OptimizedConfig, cache getTickets) echo.HandlerFunc {
	return func(ctx echo.Context) error {
		selfAlias, dsts, err := getTicketHandlerV2Args(ctx.Request())
		if err != nil {
			return &errs.InvalidParam{Message: err.Error()}
		}

		client := cfg.FindClientByAlias(selfAlias)
		if client == nil {
			return &errs.InvalidParam{
				Message: fmt.Sprintf("couldn't find client in config by alias: '%s'", selfAlias),
			}
		}

		updateTime := cache.GetTicketUpdateTime()
		response := make(TicketsResponse, len(dsts))
		for _, dstAlias := range dsts {
			dst, found := client.Dsts[dstAlias]
			if !found {
				return &errs.InvalidParam{
					Message: fmt.Sprintf("can't find in config destination tvmid for src = %s, dstparam = %s",
						selfAlias, dstAlias,
					),
				}
			}

			ticket, er, err := cache.GetTicket(client.SelfTvmID, dst.ID)
			if err != nil {
				return &errs.InvalidParam{Message: err.Error()}
			}

			response[dstAlias] = TicketStruct{Ticket: ticket, ID: dst.ID, Error: er}
		}

		ctx.Response().Header().Set(birthTimeHeader, strconv.FormatInt(updateTime.Unix(), 10))
		return ctx.JSON(http.StatusOK, response)
	}
}
